<?php
// 1. Cargar la base de datos
require_once "config/db.php"; 

// 2. Verificar inmediatamente si la conexión existe
if (!isset($pdo) || $pdo === null) {
    die("Error crítico: No se pudo establecer la conexión con la base de datos. Revisa config/db.php");
}

try {
    // 3. Ejecutar la consulta de productos
    $sqlProducts = "SELECT p.*, c.name as sub_cat, c.id as sub_cat_id, parent.name as main_cat, b.name as brand_name 
                    FROM products p 
                    JOIN categories c ON p.category_id = c.id 
                    LEFT JOIN categories parent ON c.parent_id = parent.id 
                    LEFT JOIN brands b ON p.brand_id = b.id
                    WHERE p.status = 'active' 
                    ORDER BY p.created_at DESC";

    $stmtProd = $pdo->query($sqlProducts); 
    $productsDB = $stmtProd->fetchAll(PDO::FETCH_ASSOC);

    // ==========================================

    // 2. EQUIPO (Escuadrón JFD)

    // ==========================================

    $team = [];

    try {
        // Usamos una variable distinta ($stmtTeam) para no mezclar

        $stmtTeam = $pdo->query("SELECT * FROM team_members ORDER BY id ASC");

        $team = $stmtTeam->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        $team = [];
    }

    // ==========================================

    // 3. CATEGORÍAS

    // ==========================================

    $stmtCat = $pdo->query("SELECT * FROM categories ORDER BY name ASC");

    $categories = $stmtCat->fetchAll(PDO::FETCH_ASSOC);

    // ==========================================

    // 4. BANNERS

    // ==========================================

    $banners = [];

    try {
        $stmtBan = $pdo->query(
            "SELECT * FROM banners WHERE is_active = 1 ORDER BY display_order ASC"
        );

        $banners = $stmtBan->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        // Fallback silencioso
    }

// ==========================================
    // 5. PROCESAR DATOS PARA JS
    // ==========================================

    $productsJS = [];
    $categoriesJS = [];

    foreach ($productsDB as $row) {
        // Determinar Categoría Principal y Subcategoría
        $main = $row["main_cat"] ? $row["main_cat"] : $row["sub_cat"];
        $sub = $row["sub_cat"];

        // Construir árbol de categorías
        $categoriesJS[$main][] = $sub;

        // Construir array de productos limpio
        $productsJS[] = [
            "id" => $row["id"],
            "sku" => $row["sku"] ?? "SIN-SKU",
            "name" => $row["name"],
            "description" => $row["description"] ?? "Producto de audio profesional.",
            "short_description" => $row["short_description"] ?? "",
            "price" => (float) $row["price"],
            "stock" => (int) ($row["stock_quantity"] ?? 0),
            "brand" => $row["brand_name"] ?? "JFD",
            "category" => $main,
            "category_id" => $row["category_id"],
            "sub_cat" => $sub,
            "specifications" => $row["specifications"] ?? null,
            
            // --- AQUÍ ESTÁ EL CAMBIO DE RUTA ---
            "image" => $row["main_image_url"]
                ? "public/img/productos/" . $row["main_image_url"] 
                : "public/img/no-image.png",
        ];
    }

    // Limpiar duplicados en categorías
    foreach ($categoriesJS as $k => $v) {
        $categoriesJS[$k] = array_values(array_unique($v));
        // Asegurar que "Todo" esté al inicio
        if (!in_array("Todo", $categoriesJS[$k])) {
            array_unshift($categoriesJS[$k], "Todo");
        }
    }
} catch (PDOException $e) {
    die("Error crítico en la base de datos: " . $e->getMessage());
} ?>

<!DOCTYPE html>

<html lang="es" class="scroll-smooth">

<head>

    <meta charset="UTF-8">

    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <title>JFD Importaciones | Audio Profesional Perú</title>

    <link rel="icon" type="image/png" href="public/img/logo-jfd.png">

    

    <script src="https://cdn.tailwindcss.com"></script>

    <link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css" rel="stylesheet"/>

    <link href="https://fonts.googleapis.com/css2?family=Barlow:ital,wght@0,400;0,600;0,800;1,900&family=Roboto:wght@300;400;500&display=swap" rel="stylesheet">




    <script>

        tailwind.config = {

            darkMode: 'class',

            theme: {

                extend: {

                    fontFamily: {

                        sans: ['Roboto', 'sans-serif'],

                        display: ['Barlow', 'sans-serif'],

                    },

                    colors: {

                        brand: {

                            black: '#050505',

                            dark: '#121212',

                            red: '#D90429',

                            red_hover: '#EF233C',

                            yellow: '#FFD60A',

                            light: '#F8F9FA'

                        }

                    },

                    boxShadow: {

                        'neon': '0 0 25px rgba(217, 4, 41, 0.6)',

                        'card': '0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)',

                    },

                    animation: {

                        'float-slow': 'float 6s ease-in-out infinite',

                        'pulse-slow': 'pulse 4s cubic-bezier(0.4, 0, 0.6, 1) infinite',

                        'spin-slow': 'spin 12s linear infinite',

                    },

                    keyframes: {

                        float: {

                            '0%, 100%': { transform: 'translateY(0)' },

                            '50%': { transform: 'translateY(-20px)' },

                        }

                    }

                }

            }

        }

    </script>

    <style>

        .glass-nav {

            background: rgba(255, 255, 255, 0.92);

            backdrop-filter: blur(16px);

            -webkit-backdrop-filter: blur(16px);

            border-bottom: 1px solid rgba(0,0,0,0.08);

            box-shadow: 0 4px 30px rgba(0, 0, 0, 0.05);

        }

        .dark .glass-nav {

            background: rgba(5, 5, 5, 0.92);

            border-bottom: 1px solid rgba(255,255,255,0.08);

            box-shadow: 0 4px 30px rgba(255, 255, 255, 0.02);

        }

        .hide-scrollbar::-webkit-scrollbar { display: none; }

        body { 

            transition: background-color 0.3s ease, color 0.3s ease;

            margin: 0;

            padding: 0;

        }

        html {

            margin: 0;

            padding: 0;

        }

        

        .text-stroke-red {

            -webkit-text-stroke: 2px #D90429;

            color: transparent;

        }

        .dark .text-stroke-red {

             -webkit-text-stroke: 2px #D90429;

             color: transparent;

        }

        .bg-grid-pattern {
            background-image: linear-gradient(to right, rgba(255,255,255,0.05) 1px, transparent 1px), linear-gradient(to bottom, rgba(255,255,255,0.05) 1px, transparent 1px);
            background-size: 40px 40px;
        }

        /* Hero altura completa - EXACTO 100vh */
        .hero-full-height {
            height: 100vh;
            min-height: 600px;
        }

        /* Blog altura completa */
        .blog-full-height {
            min-height: 100vh;
        }

        /* Ajuste para compensar navbar fijo */
        @media (min-width: 768px) {
            .hero-content-wrapper {
                padding-top: 0;
                padding-bottom: 0;
            }
        }



        /* Estilos para los mapas Leaflet */

        .leaflet-container {

            height: 100%;

            border-radius: 0.5rem;

        }

        

        /* Animación suave para botones flotantes */

        .floating-button {

            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);

        }

        .custom-scrollbar::-webkit-scrollbar {

    width: 6px;

}

.custom-scrollbar::-webkit-scrollbar-track {

    background: transparent;

}

.custom-scrollbar::-webkit-scrollbar-thumb {

    background: #222;

    border-radius: 10px;

}

.dark .custom-scrollbar::-webkit-scrollbar-thumb {

    background: #D90429; /* Un toque rojo en modo oscuro */

}

    </style>

</head>



<body class="bg-brand-light text-gray-900 dark:bg-brand-black dark:text-gray-100 antialiased selection:bg-brand-red selection:text-white">



    <!-- NAVBAR MEJORADO - Más limpio y profesional -->

    <header class="fixed w-full top-0 z-50 glass-nav transition-all duration-300">

        <div class="container mx-auto px-4 sm:px-6 lg:px-8">

            <div class="flex items-center justify-between h-20">

                <a href="#" class="flex items-center gap-3 group">

                    <div class="relative">

                        <div class="absolute inset-0 bg-brand-red blur-lg opacity-0 group-hover:opacity-40 transition-opacity duration-500"></div>



                        <img id="header-logo" 

                             src="public/img/logo-jfd.png" 

                             alt="JFD Logo" 

                             class="relative w-12 h-12 object-contain transition-all duration-500 group-hover:scale-110 group-hover:rotate-[5deg]">

                    </div>



                    <div class="flex flex-col leading-none">

                        <span class="text-[9px] uppercase tracking-widest text-gray-400 font-semibold">Audio Profesional</span>

                    </div>

                </a>



                <!-- Nav principal - Más elegante -->

                <nav class="hidden lg:flex items-center space-x-1 font-display font-semibold text-base uppercase tracking-wide">

                    <a href="#inicio" class="px-4 py-2 rounded-md hover:bg-brand-red/5 hover:text-brand-red transition-all relative group">

                        <span>Inicio</span>

                        <div class="absolute bottom-0 left-0 w-full h-0.5 bg-brand-red scale-x-0 group-hover:scale-x-100 transition-transform"></div>

                    </a>

                    <a href="#catalogo" class="px-4 py-2 rounded-md hover:bg-brand-red/5 hover:text-brand-red transition-all relative group">

                        <span>Catálogo</span>

                        <div class="absolute bottom-0 left-0 w-full h-0.5 bg-brand-red scale-x-0 group-hover:scale-x-100 transition-transform"></div>

                    </a>

                    <a href="#nosotros" class="px-4 py-2 rounded-md hover:bg-brand-red/5 hover:text-brand-red transition-all relative group">

                        <span>Nosotros</span>

                        <div class="absolute bottom-0 left-0 w-full h-0.5 bg-brand-red scale-x-0 group-hover:scale-x-100 transition-transform"></div>

                    </a>

                    <a href="#guias" class="px-4 py-2 rounded-md hover:bg-brand-red/5 hover:text-brand-red transition-all relative group">

                        <span>Blog</span>

                        <div class="absolute bottom-0 left-0 w-full h-0.5 bg-brand-red scale-x-0 group-hover:scale-x-100 transition-transform"></div>

                    </a>

                    <a href="#contacto" class="px-4 py-2 rounded-md hover:bg-brand-red/5 hover:text-brand-red transition-all relative group">

                        <span>Sedes</span>

                        <div class="absolute bottom-0 left-0 w-full h-0.5 bg-brand-red scale-x-0 group-hover:scale-x-100 transition-transform"></div>

                    </a>

                </nav>



                <!-- Acciones del navbar -->

                <div class="flex items-center gap-2 sm:gap-3">

                    <button id="theme-btn" class="p-2.5 text-xl text-gray-500 hover:text-brand-yellow transition-all rounded-lg hover:bg-gray-100 dark:hover:bg-white/5 active:scale-95">

                        <i class="ri-moon-line dark:hidden"></i>

                        <i class="ri-sun-line hidden dark:block"></i>

                    </button>

                    <a href="admin/login.php" class="hidden sm:flex p-2.5 text-gray-500 hover:text-brand-red transition-all rounded-lg hover:bg-gray-100 dark:hover:bg-white/5" title="Zona Admin">

                        <i class="ri-user-settings-line text-xl"></i>

                    </a>

                    <button onclick="toggleCart()" class="relative group">

                        <div class="w-11 h-11 bg-gradient-to-br from-gray-100 to-gray-50 dark:from-white/10 dark:to-white/5 rounded-lg flex items-center justify-center text-brand-red group-hover:from-brand-red group-hover:to-brand-red_hover group-hover:text-white transition-all duration-300 group-hover:shadow-lg group-hover:shadow-brand-red/30 active:scale-95">

                            <i class="ri-shopping-bag-3-fill text-lg"></i>

                        </div>

                        <span id="cart-badge" class="absolute -top-1.5 -right-1.5 bg-brand-yellow text-brand-black text-[10px] font-extrabold w-5 h-5 flex items-center justify-center rounded-full scale-0 transition-transform duration-300 shadow-lg">0</span>

                    </button>

                    <button onclick="toggleMenu()" class="lg:hidden p-2 text-2xl text-gray-900 dark:text-white hover:bg-gray-100 dark:hover:bg-white/5 rounded-lg transition-all active:scale-95">

                        <i class="ri-menu-4-line"></i>

                    </button>

                </div>

            </div>

        </div>

    </header>



    <div id="mobile-menu" class="fixed inset-0 bg-brand-black/95 z-[60] transform translate-x-full transition-transform duration-300 flex flex-col justify-center items-center gap-8 backdrop-blur-sm">

        <button onclick="toggleMenu()" class="absolute top-6 right-6 text-brand-red text-4xl"><i class="ri-close-fill"></i></button>

        <a href="#inicio" onclick="toggleMenu()" class="font-display font-bold text-4xl text-white hover:text-brand-red tracking-wider">INICIO</a>

        <a href="#catalogo" onclick="toggleMenu()" class="font-display font-bold text-4xl text-white hover:text-brand-red tracking-wider">CATÁLOGO</a>

        <a href="#nosotros" onclick="toggleMenu()" class="font-display font-bold text-4xl text-white hover:text-brand-red tracking-wider">NOSOTROS</a>

        <a href="#contacto" onclick="toggleMenu()" class="font-display font-bold text-4xl text-white hover:text-brand-red tracking-wider">CONTACTO</a>

    </div>



    <!-- HERO SECTION - 100% altura de pantalla EXACTA -->

<section id="inicio" class="relative hero-full-height flex items-center justify-center overflow-hidden bg-brand-black">
    
    <div class="absolute inset-0 z-0 overflow-hidden">
        <div class="absolute inset-0 bg-grid-pattern opacity-20"></div>
        <div class="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[800px] h-[800px] bg-brand-red/20 rounded-full blur-[120px] animate-pulse-slow"></div>
        <div class="absolute bottom-0 right-0 w-[600px] h-[600px] bg-purple-900/20 rounded-full blur-[150px] mix-blend-screen"></div>
    </div>

    <div class="container mx-auto px-4 relative z-10 grid grid-cols-1 lg:grid-cols-5 gap-12 items-center pt-20">
        
        <div class="lg:col-span-2 space-y-10 animate-fade-in-up relative z-20">
            
            <div class="inline-flex items-center gap-3 px-5 py-2 bg-gradient-to-r from-brand-red/20 to-transparent border-l-4 border-brand-red rounded-r-full">
                <span class="relative flex h-3 w-3">
                  <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-brand-red opacity-75"></span>
                  <span class="relative inline-flex rounded-full h-3 w-3 bg-brand-red"></span>
                </span>
                <span class="text-white font-display font-bold uppercase tracking-[0.2em] text-[10px]">
                    Distribuidor Oficial Perú <span class="text-brand-red">| JFD PRO AUDIO</span>
                </span>
            </div>

            <div class="space-y-2">
                <h1 class="font-display font-black text-6xl md:text-7xl lg:text-8xl text-white leading-[0.8] tracking-tighter italic">
                    SONIDO <br>
                    <span class="text-stroke-red relative z-10">BRUTAL.</span>
                </h1>
                <p class="text-brand-yellow font-display font-bold text-sm tracking-widest uppercase mt-4">
                    Ingeniería Acústica de Alto Rendimiento
                </p>
            </div>

            <p class="text-gray-400 text-lg max-w-xl font-light leading-relaxed border-l-2 border-brand-red/50 pl-6">
                Importación directa de tecnología de élite para Huánuco y todo el Perú. Equipamos tus proyectos con el estándar más alto de fidelidad.
            </p>
            
            <div class="flex flex-wrap gap-5">
                <a href="#catalogo" class="group relative bg-brand-red text-white px-8 py-5 rounded-sm font-bold font-display text-lg uppercase tracking-widest overflow-hidden transition-all hover:shadow-neon hover:-translate-y-1">
                    <span class="relative z-10 flex items-center gap-2">Explorar Arsenal <i class="ri-arrow-right-line transition-transform group-hover:translate-x-2"></i></span>
                    <div class="absolute inset-0 h-full w-full scale-0 rounded-sm transition-all duration-300 group-hover:scale-100 group-hover:bg-white/10"></div>
                </a>
                <a href="https://wa.me/51989928670" target="_blank" class="group flex items-center gap-2 border-2 border-white/20 text-white px-8 py-5 rounded-sm font-bold font-display text-lg uppercase tracking-widest transition-all hover:border-brand-yellow hover:text-brand-yellow">
                    <i class="ri-customer-service-2-fill text-2xl text-brand-yellow"></i> Asesoría
                </a>
            </div>

            <div class="grid grid-cols-2 gap-8 pt-8 border-t border-white/5">
                <div class="flex items-center gap-3">
                    <i class="ri-shield-check-line text-3xl text-brand-red"></i>
                    <div>
                        <p class="text-white font-bold text-sm leading-none">GARANTÍA REAL</p>
                        <p class="text-gray-500 text-[11px] uppercase mt-1">Soporte Técnico Local</p>
                    </div>
                </div>
                <div class="flex items-center gap-3">
                    <i class="ri-truck-line text-3xl text-brand-red"></i>
                    <div>
                        <p class="text-white font-bold text-sm leading-none">ENVÍO SEGURO</p>
                        <p class="text-gray-500 text-[11px] uppercase mt-1">Cobertura Nacional</p>
                    </div>
                </div>
            </div>
        </div>

        <div class="lg:col-span-3 relative z-10 hidden lg:block">
            <div class="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[700px] h-[700px] border-2 border-dashed border-brand-red/20 rounded-full animate-spin-slow"></div>
            
            <img src="public/img/imagen-inicio.png" 
                 class="w-full max-w-none mx-auto animate-float-slow drop-shadow-[0_45px_45px_rgba(217,4,41,0.4)] transform scale-[1.3] lg:scale-[1.5] hover:scale-[1.6] transition-transform duration-700" 
                 alt="Equipo Destacado JFD"
                 onerror="this.src='public/img/logo-jfd.png'">
        </div>
    </div>
</section>



    <section id="catalogo" class="py-24 bg-white dark:bg-brand-black relative z-10">

    <div class="container mx-auto px-4">

        

        <div class="flex flex-col md:flex-row justify-between items-end mb-12 border-b border-gray-200 dark:border-white/10 pb-6">

            <div>

                <span class="text-brand-red font-bold uppercase tracking-widest text-xs mb-2 block">Explora nuestros productos</span>

                <h2 class="font-display font-black text-4xl md:text-5xl text-gray-900 dark:text-white uppercase">

                    Catálogo <span class="text-gray-400 dark:text-gray-600"></span>

                </h2>

            </div>

            

            <div class="mt-6 md:mt-0 flex items-center gap-4">

                <label class="text-xs font-bold uppercase text-gray-500 tracking-wider">Ordenar por:</label>

                <select id="sort-select" onchange="applyFilters()" class="bg-gray-100 dark:bg-white/5 border border-transparent focus:border-brand-red text-gray-900 dark:text-white text-sm font-bold uppercase tracking-wide py-3 px-4 rounded outline-none transition-all cursor-pointer">

                    <option value="newest">Lo más nuevo</option>

                    <option value="price-asc">Precio: Bajo a Alto</option>

                    <option value="price-desc">Precio: Alto a Bajo</option>

                    <option value="name-asc">Nombre: A - Z</option>

                </select>

            </div>

        </div>



        <div class="flex flex-col lg:flex-row gap-12">

            

            <aside class="w-full lg:w-1/4 space-y-12">

                <div class="relative group">

                    <input type="text" id="search-input" onkeyup="applyFilters()" placeholder="BUSCAR EQUIPO..." 

                        class="w-full bg-transparent border-b-2 border-gray-300 dark:border-gray-700 focus:border-brand-red text-gray-900 dark:text-white font-bold uppercase tracking-wider py-3 pr-10 outline-none transition-colors placeholder-gray-400">

                    <i class="ri-search-line absolute right-0 top-3 text-xl text-gray-400 group-focus-within:text-brand-red transition-colors"></i>

                </div>



                <div>

                    <h3 class="font-display font-bold text-lg text-gray-900 dark:text-white uppercase mb-6 flex items-center gap-2">

                        <i class="ri-layout-grid-line text-brand-red"></i> Categorías

                    </h3>

                    <ul class="space-y-3" id="category-filters">

                        <li>

                            <label class="flex items-center gap-3 cursor-pointer group">

                                <input type="checkbox" value="Audio Profesional" class="filter-category w-4 h-4 accent-brand-red bg-gray-200 border-none rounded cursor-pointer" onchange="applyFilters()">

                                <span class="text-sm font-medium text-gray-500 group-hover:text-brand-red transition-colors uppercase tracking-wide">Audio Profesional</span>

                            </label>

                        </li>

                        <li>

                            <label class="flex items-center gap-3 cursor-pointer group">

                                <input type="checkbox" value="Instrumentos" class="filter-category w-4 h-4 accent-brand-red" onchange="applyFilters()">

                                <span class="text-sm font-medium text-gray-500 group-hover:text-brand-red transition-colors uppercase tracking-wide">Instrumentos</span>

                            </label>

                        </li>

                        </ul>

                </div>



                <div>

                    <h3 class="font-display font-bold text-lg text-gray-900 dark:text-white uppercase mb-6 flex items-center gap-2">

                        <i class="ri-money-dollar-circle-line text-brand-red"></i> Rango de Precio

                    </h3>

                    <div class="flex items-center gap-4 mb-4">

                        <input type="number" id="min-price" placeholder="Min" class="w-full bg-gray-100 dark:bg-white/5 border border-gray-200 dark:border-white/10 rounded p-2 text-sm text-center font-bold text-gray-900 dark:text-white outline-none focus:border-brand-red transition-colors">

                        <span class="text-gray-400">-</span>

                        <input type="number" id="max-price" placeholder="Max" class="w-full bg-gray-100 dark:bg-white/5 border border-gray-200 dark:border-white/10 rounded p-2 text-sm text-center font-bold text-gray-900 dark:text-white outline-none focus:border-brand-red transition-colors">

                    </div>

                    <button onclick="applyFilters()" class="w-full bg-gray-900 dark:bg-white/10 hover:bg-brand-red text-white text-xs font-bold uppercase tracking-widest py-3 rounded transition-colors">

                        Filtrar Precio

                    </button>

                </div>

            </aside>



            <div class="w-full lg:w-3/4">

                

                <div id="products-grid" class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-8">

                    </div>



                <div id="no-results" class="hidden text-center py-20">

                    <i class="ri-emotion-unhappy-line text-6xl text-gray-300 dark:text-white/20 mb-4 block"></i>

                    <p class="text-xl font-display font-bold text-gray-500 uppercase">No encontramos equipos con esos filtros</p>

                    <button onclick="resetFilters()" class="mt-4 text-brand-red font-bold underline uppercase tracking-wider text-sm">Limpiar Filtros</button>

                </div>



                <div class="mt-20 flex justify-center border-t border-gray-200 dark:border-white/10 pt-10">

                    <div id="pagination" class="flex items-center gap-2">

                        </div>

                </div>



            </div>

        </div>

    </div>

</section>



    <section id="nosotros" class="py-24 bg-white dark:bg-brand-black transition-colors relative z-20 border-t border-gray-100 dark:border-white/5">

        <div class="container mx-auto px-4 grid grid-cols-1 lg:grid-cols-2 gap-20 items-center">

            <div class="relative group">

                <div class="absolute -top-4 -left-4 w-full h-full border-2 border-brand-red rounded-sm z-0"></div>

                <div class="absolute -bottom-4 -right-4 w-full h-full bg-brand-yellow/10 rounded-sm z-0"></div>

                <img src="https://images.unsplash.com/photo-1598653222000-6b7b7a552625?q=80&w=800" 

                     class="relative z-10 w-full h-[500px] object-cover grayscale group-hover:grayscale-0 transition-all duration-700 shadow-2xl" alt="Estudio JFD">

            </div>

            <div>

                <span class="text-brand-yellow font-bold uppercase tracking-widest text-sm mb-4 block"><i class="ri-medal-line"></i> Excelencia desde 2016</span>

                <h2 class="font-display font-black text-5xl lg:text-6xl text-gray-900 dark:text-white mb-8 leading-[0.9]">

                    INGENIERÍA <br> AL SERVICIO DEL <span class="text-brand-red">ARTE</span>

                </h2>

                <div class="space-y-6 text-lg text-gray-600 dark:text-gray-300 leading-relaxed font-light">

                    <p>

                        En <strong>JFD Importaciones</strong>, no movemos cajas, potenciamos talentos. Entendemos que un fader que falla en vivo no es una opción. Por eso, cada equipo que importamos a Huánuco ha sido seleccionado por su fiabilidad en el campo de batalla.

                    </p>

                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-6 mt-8">

                        <div class="bg-gray-50 dark:bg-white/5 p-6 border-l-4 border-brand-red">

                            <h4 class="font-bold text-xl text-gray-900 dark:text-white font-display mb-2">MISIÓN</h4>

                            <p class="text-sm">Proveer tecnología acústica de clase mundial con un respaldo técnico local inigualable.</p>

                        </div>

                        <div class="bg-gray-50 dark:bg-white/5 p-6 border-l-4 border-brand-yellow">

                            <h4 class="font-bold text-xl text-gray-900 dark:text-white font-display mb-2">GARANTÍA JFD</h4>

                            <p class="text-sm">Soporte directo, repuestos originales y asesoría de ingenieros reales.</p>

                        </div>

                    </div>

                </div>

            </div>

        </div>

    </section>



<section id="equipo" class="py-12 bg-brand-light dark:bg-brand-dark relative z-20">
    <div class="container mx-auto px-4">
        <div class="text-center mb-8">
            <h2 class="font-display font-black text-3xl text-gray-900 dark:text-white uppercase">El Escuadrón JFD</h2>
            
            <p class="text-gray-500 dark:text-gray-400 mt-1 text-base">
                <span class="text-brand-red font-bold"><?php echo count($team); ?></span> Expertos. Una sola pasión.
            </p>
        </div>

        <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4">
            <?php if (!empty($team)): ?>
                <?php foreach ($team as $member): ?>
                    <?php
                    // --- DATOS INDIVIDUALES ---
                    $nombre = htmlspecialchars($member["full_name"]);
                    $cargo = htmlspecialchars($member["position"]);

                    // 1. RUTA DE IMAGEN ACTUALIZADA
                    // Busca en public/img/asesores/ usando el nombre guardado en BD
                    $photoSrc = !empty($member["photo_url"])
                        ? "public/img/asesores/" . $member["photo_url"]
                        : "public/img/logo-jfd.png"; // Imagen por defecto si campo está vacío

                    // 2. LIMPIEZA DE TELÉFONO
                    $telefonoDB = $member["whatsapp"];
                    $telefonoClean = preg_replace("/[^0-9]/", "", $telefonoDB);
                    
                    if (empty($telefonoClean)) {
                        $telefonoClean = "51989928670"; // Central por defecto
                    }

                    // 3. EMAIL
                    $emailDB = $member["email"];
                    if (empty($emailDB)) {
                        $emailDB = "jfdimportacionesperu@gmail.com";
                    }

                    // Enlaces
                    $waLink = "https://wa.me/{$telefonoClean}?text=" . urlencode("Hola {$nombre}, vi tu perfil en la web y deseo asesoría.");
                    $emailLink = "mailto:{$emailDB}?subject=Asesoria JFD - Atn: {$nombre}";
                    ?>

                    <div class="group relative bg-white dark:bg-brand-dark border border-gray-200 dark:border-white/10 hover:border-brand-red transition-all duration-300 shadow-sm hover:shadow-md overflow-hidden">
                        
                        <div class="aspect-[3/2] relative overflow-hidden bg-gray-50 dark:bg-gray-800 bg-center bg-no-repeat"
                             style="background-image: url('public/img/logo-jfd.png'); background-size: 40%; background-position: center;">
                            
                            <img src="<?php echo $photoSrc; ?>" 
                                 alt="<?php echo $nombre; ?>" 
                                 onerror="this.style.opacity='0'" 
                                 class="absolute inset-0 w-full h-full object-cover filter grayscale group-hover:grayscale-0 group-hover:scale-105 transition-all duration-500 z-10">
                            
                            <div class="absolute inset-0 bg-brand-dark/90 backdrop-blur-[2px] opacity-0 group-hover:opacity-100 transition-all duration-300 flex items-center justify-center gap-3 z-20">
                                <a href="<?php echo $waLink; ?>" target="_blank" title="WhatsApp de <?php echo $nombre; ?>" class="w-10 h-10 bg-green-500 rounded text-white flex items-center justify-center hover:scale-110 transition-transform shadow-lg">
                                    <i class="ri-whatsapp-fill text-xl"></i>
                                </a>
                                <a href="<?php echo $emailLink; ?>" title="Email de <?php echo $nombre; ?>" class="w-10 h-10 bg-blue-600 rounded text-white flex items-center justify-center hover:scale-110 transition-transform shadow-lg">
                                    <i class="ri-mail-fill text-xl"></i>
                                </a>
                            </div>
                        </div>

                        <div class="h-24 px-3 flex flex-col justify-center text-center bg-white dark:bg-brand-dark relative z-30 border-t border-gray-100 dark:border-white/5">
                            <h3 class="font-display font-bold text-sm md:text-base text-gray-900 dark:text-white uppercase leading-tight line-clamp-2 h-10 flex items-center justify-center">
                                <?php echo $nombre; ?>
                            </h3>
                            
                            <div class="w-8 h-0.5 bg-brand-red mx-auto my-2"></div>
                            
                            <p class="text-xs font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                <?php echo $cargo; ?>
                            </p>
                        </div>
                    </div>

                <?php endforeach; ?>
            <?php else: ?>
                <div class="col-span-full text-center py-8 text-gray-400">Cargando escuadrón...</div>
            <?php endif; ?>
        </div>
    </div>
</section>



<section id="guias" class="blog-full-height flex items-center bg-brand-black text-white relative z-20 overflow-hidden">

    <div class="absolute inset-0 bg-[url('https://www.transparenttextures.com/patterns/carbon-fibre.png')] opacity-20 mix-blend-overlay"></div>

    <div class="absolute top-0 left-1/2 -translate-x-1/2 w-full h-1 bg-gradient-to-r from-transparent via-brand-red to-transparent"></div>

    

    <div class="container mx-auto px-4 py-20 relative z-10">

        <div class="flex flex-col md:flex-row justify-between items-end mb-12 gap-6">

            <div>

                <h2 class="font-display font-black text-4xl md:text-5xl uppercase leading-none">Academia <span class="text-brand-yellow">JFD</span></h2>

                <p class="text-gray-400 mt-4 text-base md:text-lg">Conocimiento técnico para tomar mejores decisiones.</p>

            </div>
        </div>



        <div class="grid grid-cols-2 lg:grid-cols-5 gap-4 justify-center">

            

            <article class="bg-brand-dark border border-white/10 hover:border-brand-yellow transition-colors group rounded-sm overflow-hidden shadow-lg hover:shadow-brand-yellow/20 cursor-pointer" onclick="openArticleModal('potencia-espacio')">

                <div class="h-48 overflow-hidden relative">

                    <div class="absolute inset-0 bg-brand-black/30 group-hover:bg-transparent transition-colors z-10"></div>

                    <img src="https://images.unsplash.com/photo-1574169208507-84376144848b?q=80&w=800" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700" alt="Parlantes Potentes">

                    <div class="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500 z-20">

                        <div class="relative w-14 h-14 bg-brand-yellow rounded-full flex items-center justify-center shadow-2xl">

                            <i class="ri-book-open-line text-2xl text-brand-black"></i>

                        </div>

                    </div>

                </div>

                <div class="p-6">

                    <span class="inline-block bg-brand-yellow/10 text-brand-yellow text-[10px] font-bold uppercase px-2 py-1 rounded-full mb-3">Guía Técnica</span>

                    <h3 class="font-display font-bold text-lg mb-3 leading-tight group-hover:text-brand-yellow transition-colors line-clamp-2">¿QUÉ POTENCIA NECESITAS?</h3>

                    <div class="inline-flex items-center gap-1 text-white font-bold font-display uppercase tracking-wider hover:text-brand-yellow transition-colors text-xs">

                        Leer <i class="ri-arrow-right-s-line text-base"></i>

                    </div>

                </div>

            </article>

            

            <article class="bg-brand-dark border border-white/10 hover:border-brand-yellow transition-colors group rounded-sm overflow-hidden shadow-lg hover:shadow-brand-yellow/20 cursor-pointer" onclick="openArticleModal('primer-equipo')">

                <div class="h-48 overflow-hidden relative">

                    <div class="absolute inset-0 bg-brand-black/30 group-hover:bg-transparent transition-colors z-10"></div>

                    <img src="https://images.unsplash.com/photo-1511671782779-c97d3d27a1d4?q=80&w=800" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700" alt="Primer Equipo">

                    <div class="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500 z-20">

                        <div class="relative w-14 h-14 bg-brand-yellow rounded-full flex items-center justify-center shadow-2xl">

                            <i class="ri-book-open-line text-2xl text-brand-black"></i>

                        </div>

                    </div>

                </div>

                <div class="p-6">

                    <span class="inline-block bg-green-500/10 text-green-500 text-[10px] font-bold uppercase px-2 py-1 rounded-full mb-3">Compra Inteligente</span>

                    <h3 class="font-display font-bold text-lg mb-3 leading-tight group-hover:text-brand-yellow transition-colors line-clamp-2">TU PRIMER EQUIPO</h3>

                    <div class="inline-flex items-center gap-1 text-white font-bold font-display uppercase tracking-wider hover:text-brand-yellow transition-colors text-xs">

                        Leer <i class="ri-arrow-right-s-line text-base"></i>

                    </div>

                </div>

            </article>



            <article class="bg-brand-dark border border-white/10 hover:border-brand-yellow transition-colors group rounded-sm overflow-hidden shadow-lg hover:shadow-brand-yellow/20 cursor-pointer" onclick="openArticleModal('casa-vs-eventos')">

                <div class="h-48 overflow-hidden relative">

                    <div class="absolute inset-0 bg-brand-black/30 group-hover:bg-transparent transition-colors z-10"></div>

                    <img src="https://images.unsplash.com/photo-1492684223066-81342ee5ff30?q=80&w=800" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700" alt="Evento en Vivo">

                    <div class="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500 z-20">

                        <div class="relative w-14 h-14 bg-brand-yellow rounded-full flex items-center justify-center shadow-2xl">

                            <i class="ri-book-open-line text-2xl text-brand-black"></i>

                        </div>

                    </div>

                </div>

                <div class="p-6">

                    <span class="inline-block bg-brand-red/10 text-brand-red text-[10px] font-bold uppercase px-2 py-1 rounded-full mb-3">Comparativa</span>

                    <h3 class="font-display font-bold text-lg mb-3 leading-tight group-hover:text-brand-yellow transition-colors line-clamp-2">CASA VS. EVENTOS</h3>

                    <div class="inline-flex items-center gap-1 text-white font-bold font-display uppercase tracking-wider hover:text-brand-yellow transition-colors text-xs">

                        Leer <i class="ri-arrow-right-s-line text-base"></i>

                    </div>

                </div>

            </article>



            <article class="bg-brand-dark border border-white/10 hover:border-brand-yellow transition-colors group rounded-sm overflow-hidden shadow-lg hover:shadow-brand-yellow/20 cursor-pointer" onclick="openArticleModal('guia-conexiones')">

                <div class="h-48 overflow-hidden relative">

                    <div class="absolute inset-0 bg-brand-black/30 group-hover:bg-transparent transition-colors z-10"></div>

                    <img src="https://png.pngtree.com/thumb_back/fw800/background/20251210/pngtree-close-up-of-a-modular-synthesizer-with-colorful-patch-cables-connecting-image_20786178.webp" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700" alt="Cables de Audio Profesionales">

                    <div class="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500 z-20">

                        <div class="relative w-14 h-14 bg-brand-yellow rounded-full flex items-center justify-center shadow-2xl">

                            <i class="ri-book-open-line text-2xl text-brand-black"></i>

                        </div>

                    </div>

                </div>

                <div class="p-6">

                    <span class="inline-block bg-blue-500/10 text-blue-500 text-[10px] font-bold uppercase px-2 py-1 rounded-full mb-3">Principiantes</span>

                    <h3 class="font-display font-bold text-lg mb-3 leading-tight group-hover:text-brand-yellow transition-colors line-clamp-2">GUÍA DE CABLES</h3>

                    <div class="inline-flex items-center gap-1 text-white font-bold font-display uppercase tracking-wider hover:text-brand-yellow transition-colors text-xs">

                        Leer <i class="ri-arrow-right-s-line text-base"></i>

                    </div>

                </div>

            </article>



            <article class="bg-brand-dark border border-white/10 hover:border-brand-yellow transition-colors group rounded-sm overflow-hidden shadow-lg hover:shadow-brand-yellow/20 cursor-pointer" onclick="openArticleModal('cuidado-parlantes')">

                <div class="h-48 overflow-hidden relative">

                    <div class="absolute inset-0 bg-brand-black/30 group-hover:bg-transparent transition-colors z-10"></div>

                    <img src="https://images.unsplash.com/photo-1520523839897-bd0b52f945a0?q=80&w=800" class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700" alt="Consola">

                    <div class="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-500 z-20">

                        <div class="relative w-14 h-14 bg-brand-yellow rounded-full flex items-center justify-center shadow-2xl">

                            <i class="ri-book-open-line text-2xl text-brand-black"></i>

                        </div>

                    </div>

                </div>

                <div class="p-6">

                    <span class="inline-block bg-orange-500/10 text-orange-500 text-[10px] font-bold uppercase px-2 py-1 rounded-full mb-3">Mantenimiento</span>

                    <h3 class="font-display font-bold text-lg mb-3 leading-tight group-hover:text-brand-yellow transition-colors line-clamp-2">CUIDA TUS PARLANTES</h3>

                    <div class="inline-flex items-center gap-1 text-white font-bold font-display uppercase tracking-wider hover:text-brand-yellow transition-colors text-xs">

                        Leer <i class="ri-arrow-right-s-line text-base"></i>

                    </div>

                </div>

            </article>



        </div>

    </div>

</section>



<section id="contacto" class="py-24 bg-brand-light dark:bg-brand-dark transition-colors relative z-20">

    <div class="container mx-auto px-4">

        <div class="text-center mb-16">

            <span class="text-brand-red font-bold uppercase tracking-widest text-sm mb-3 block">

                <i class="ri-map-pin-line"></i> Cobertura Nacional

            </span>

            <h2 class="font-display font-black text-5xl text-gray-900 dark:text-white uppercase mb-4">

                Nuestras <span class="text-brand-red">Bases</span>

            </h2>

            <p class="text-gray-500 dark:text-gray-400 text-lg max-w-2xl mx-auto">

                Presencia estratégica en las principales ciudades para brindarte soporte inmediato

            </p>

        </div>

        

        <div class="grid grid-cols-1 md:grid-cols-3 gap-8">

            <div class="group relative bg-white dark:bg-brand-black rounded-lg overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 border-2 border-transparent hover:border-brand-red">

                <div class="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-brand-red via-brand-red_hover to-brand-red"></div>

                

                <div class="h-64 bg-gray-200 rounded-t-lg overflow-hidden relative border-b-2 border-gray-100 dark:border-white/5">

                    <iframe 

                        src="https://maps.google.com/maps?q=TIENDA+MUSICAL+JFD+IMPORTACIONES+HUANUCO&t=&z=17&ie=UTF8&iwloc=&output=embed" 

                        width="100%" 

                        height="100%" 

                        style="border:0;" 

                        allowfullscreen="" 

                        loading="lazy" 

                        referrerpolicy="no-referrer-when-downgrade">

                    </iframe>

                    

                    <div class="absolute top-4 right-4 bg-brand-red text-white px-3 py-1.5 rounded-md text-xs font-bold uppercase tracking-wider shadow-lg pointer-events-none">

                        Sede Principal

                    </div>

                </div>

                

                <div class="p-6 space-y-4">

                    <div class="flex items-center gap-3">

                        <div class="w-12 h-12 bg-brand-red/10 rounded-lg flex items-center justify-center">

                            <i class="ri-building-line text-2xl text-brand-red"></i>

                        </div>

                        <div>

                            <h3 class="font-display font-bold text-2xl text-gray-900 dark:text-white">HUÁNUCO</h3>

                            <p class="text-xs text-gray-400 uppercase tracking-wider">Región Centro</p>

                        </div>

                    </div>

                    

                    <div class="space-y-2 text-sm text-gray-600 dark:text-gray-300">

                        <div class="flex items-center gap-2">

                            <i class="ri-map-pin-fill text-brand-red"></i>

                            <span>Jr. Independencia 1145</span>

                        </div>

                        <div class="flex items-center gap-2">

                            <i class="ri-phone-fill text-brand-red"></i>

                            <span>+51 989 928 670</span>

                        </div>

                        <div class="flex items-center gap-2">

                            <i class="ri-time-fill text-brand-red"></i>

                            <span>Lun-Sáb: 9:00am - 7:00pm</span>

                        </div>

                    </div>

                    

                    <a href="https://wa.me/51989928670" target="_blank" class="block w-full py-3 bg-brand-red hover:bg-brand-red_hover text-white text-center font-bold rounded-lg transition-all transform hover:scale-105">

                        <i class="ri-whatsapp-fill mr-2"></i>Contactar Sede

                    </a>

                </div>

            </div>



            <div class="group relative bg-white dark:bg-brand-black rounded-lg overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 border-2 border-transparent hover:border-brand-yellow">

                <div class="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-brand-yellow via-yellow-400 to-brand-yellow"></div>

                

                <div class="h-64 bg-gray-200 rounded-t-lg overflow-hidden relative border-b-2 border-gray-100 dark:border-white/5">

                    <iframe 

                        src="https://maps.google.com/maps?q=Jr.+San+Martin+418,+Satipo,+Peru&t=&z=16&ie=UTF8&iwloc=&output=embed" 

                        width="100%" 

                        height="100%" 

                        style="border:0;" 

                        allowfullscreen="" 

                        loading="lazy" 

                        referrerpolicy="no-referrer-when-downgrade">

                    </iframe>

                    

                    <div class="absolute top-4 right-4 bg-brand-yellow text-brand-black px-3 py-1.5 rounded-md text-xs font-bold uppercase tracking-wider shadow-lg pointer-events-none">

                        Sucursal

                    </div>

                </div>

                

                <div class="p-6 space-y-4">

                    <div class="flex items-center gap-3">

                        <div class="w-12 h-12 bg-brand-yellow/10 rounded-lg flex items-center justify-center">

                            <i class="ri-store-line text-2xl text-brand-yellow"></i>

                        </div>

                        <div>

                            <h3 class="font-display font-bold text-2xl text-gray-900 dark:text-white">SATIPO</h3>

                            <p class="text-xs text-gray-400 uppercase tracking-wider">Región Junín</p>

                        </div>

                    </div>

                    

                    <div class="space-y-2 text-sm text-gray-600 dark:text-gray-300">

                        <div class="flex items-center gap-2">

                            <i class="ri-map-pin-fill text-brand-yellow"></i>

                            <span>Jr. San Martín 418</span>

                        </div>

                        <div class="flex items-center gap-2">

                            <i class="ri-phone-fill text-brand-yellow"></i>

                            <span>+51 996 145 569</span>

                        </div>

                        <div class="flex items-center gap-2">

                            <i class="ri-time-fill text-brand-yellow"></i>

                            <span>Lun-Sáb: 9:00am - 7:00pm</span>

                        </div>

                    </div>

                    

                    <a href="https://wa.me/51996145569" target="_blank" class="block w-full py-3 bg-brand-yellow hover:bg-yellow-400 text-brand-black text-center font-bold rounded-lg transition-all transform hover:scale-105">

                        <i class="ri-whatsapp-fill mr-2"></i>Contactar Sede

                    </a>

                </div>

            </div>



            <div class="group relative bg-white dark:bg-brand-black rounded-lg overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 border-2 border-transparent hover:border-brand-red">

                <div class="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-brand-red via-brand-red_hover to-brand-red"></div>

                

                <div class="h-64 bg-gray-200 rounded-t-lg overflow-hidden relative border-b-2 border-gray-100 dark:border-white/5">

                    <iframe 

                        src="https://maps.google.com/maps?q=Centro+Comercial+El+Hueco,+Lima&t=&z=16&ie=UTF8&iwloc=&output=embed" 

                        width="100%" 

                        height="100%" 

                        style="border:0;" 

                        allowfullscreen="" 

                        loading="lazy" 

                        referrerpolicy="no-referrer-when-downgrade">

                    </iframe>



                    <div class="absolute top-4 right-4 bg-brand-red text-white px-3 py-1.5 rounded-md text-xs font-bold uppercase tracking-wider shadow-lg pointer-events-none">

                        Punto de Venta

                    </div>

                </div>

                

                <div class="p-6 space-y-4">

                    <div class="flex items-center gap-3">

                        <div class="w-12 h-12 bg-brand-red/10 rounded-lg flex items-center justify-center">

                            <i class="ri-shopping-bag-line text-2xl text-brand-red"></i>

                        </div>

                        <div>

                            <h3 class="font-display font-bold text-2xl text-gray-900 dark:text-white">LIMA</h3>

                            <p class="text-xs text-gray-400 uppercase tracking-wider">Capital</p>

                        </div>

                    </div>

                    

                    <div class="space-y-2 text-sm text-gray-600 dark:text-gray-300">

                        <div class="flex items-center gap-2">

                            <i class="ri-map-pin-fill text-brand-red"></i>

                            <span>C.C. El Hueco, Puesto 22A</span>

                        </div>

                        <div class="flex items-center gap-2">

                            <i class="ri-phone-fill text-brand-red"></i>

                            <span>+51 904 734 304</span>

                        </div>

                        <div class="flex items-center gap-2">

                            <i class="ri-time-fill text-brand-red"></i>

                            <span>Lun-Sáb: 9:00am - 7:00pm</span>

                        </div>

                    </div>

                    

                    <a href="https://wa.me/51904734304" target="_blank" class="block w-full py-3 bg-brand-red hover:bg-brand-red_hover text-white text-center font-bold rounded-lg transition-all transform hover:scale-105">

                        <i class="ri-whatsapp-fill mr-2"></i>Contactar Sede

                    </a>

                </div>

            </div>

        </div>

        

        <div class="mt-16">

            <div class="bg-white dark:bg-brand-black rounded-lg shadow-xl p-8 border border-gray-200 dark:border-white/5">

                <div class="grid grid-cols-1 md:grid-cols-4 gap-8 items-center">

                    <div class="flex items-center gap-3 text-gray-600 dark:text-gray-300">

                        <div class="w-12 h-12 bg-brand-red/10 rounded-full flex items-center justify-center">

                            <i class="ri-truck-fill text-2xl text-brand-red"></i>

                        </div>

                        <div>

                            <span class="font-bold text-gray-900 dark:text-white block text-sm">Envíos a todo el Perú</span>

                            <span class="text-xs text-gray-500">Delivery express disponible</span>

                        </div>

                    </div>

                    

                    <div class="flex items-center gap-3 text-gray-600 dark:text-gray-300">

                        <div class="w-12 h-12 bg-brand-yellow/10 rounded-full flex items-center justify-center">

                            <i class="ri-customer-service-fill text-2xl text-brand-yellow"></i>

                        </div>

                        <div>

                            <span class="font-bold text-gray-900 dark:text-white block text-sm">Soporte técnico 24/7</span>

                            <span class="text-xs text-gray-500">Atención personalizada</span>

                        </div>

                    </div>



                    <div class="flex items-center gap-3 text-gray-600 dark:text-gray-300">

                        <div class="w-12 h-12 bg-green-500/10 rounded-full flex items-center justify-center">

                            <i class="ri-shield-check-fill text-2xl text-green-500"></i>

                        </div>

                        <div>

                            <span class="font-bold text-gray-900 dark:text-white block text-sm">Garantía Oficial</span>

                            <span class="text-xs text-gray-500">12 meses en todos los equipos</span>

                        </div>

                    </div>



                    <div class="flex items-center gap-3 text-gray-600 dark:text-gray-300">

                        <div class="w-12 h-12 bg-purple-500/10 rounded-full flex items-center justify-center">

                            <i class="ri-tools-fill text-2xl text-purple-500"></i>

                        </div>

                        <div>

                            <span class="font-bold text-gray-900 dark:text-white block text-sm">Instalación Incluida</span>

                            <span class="text-xs text-gray-500">Servicio de configuración</span>

                        </div>

                    </div>

                </div>

            </div>

        </div>

    </div>

</section>



    <!-- FOOTER MEJORADO con crédito de Consultoría Quanta -->

    <footer class="bg-gradient-to-b from-brand-black to-black text-white pt-20 pb-8 relative z-20 border-t border-brand-red/20">

        <div class="absolute inset-0 bg-[url('https://www.transparenttextures.com/patterns/carbon-fibre.png')] opacity-10"></div>

        

        <div class="container mx-auto px-4 relative z-10">

            <!-- Grid principal -->

            <div class="grid grid-cols-1 md:grid-cols-12 gap-12 mb-16">

                <!-- Columna 1: Branding (40%) -->

                <div class="col-span-1 md:col-span-5 space-y-6">

                    <div class="flex items-center gap-3 mb-6">

                        <div class="relative">

                            <div class="absolute inset-0 bg-brand-red blur-lg opacity-30"></div>

                            <img src="public/img/logo-jfd.png" 

                                 alt="JFD Logo" 

                                 class="relative w-14 h-14 object-contain">

                        </div>

                        <div class="flex flex-col leading-none">

                            <span class="text-[10px] uppercase tracking-widest text-gray-400 font-semibold">Audio Profesional Perú</span>

                        </div>

                    </div>

                    <p class="text-gray-400 text-base font-light leading-relaxed max-w-md">

                        Elevando el estándar del audio profesional en Perú desde 2016. Importación directa, asesoría técnica real y soporte post-venta garantizado.

                    </p>

                    <div class="flex gap-3 pt-2">

                        <a href="https://www.facebook.com/share/1Dfc4VrWMj/" class="group w-11 h-11 bg-white/5 hover:bg-brand-red flex items-center justify-center rounded-lg border border-white/10 hover:border-brand-red transition-all duration-300 hover:shadow-lg hover:shadow-brand-red/30">

                            <i class="ri-facebook-fill text-xl text-gray-400 group-hover:text-white transition-colors"></i>

                        </a>

                        <a href="https://www.tiktok.com/@jfdimportacionesperu?_r=1&_t=ZS-93tQYNZ8emn" class="group w-11 h-11 bg-white/5 hover:bg-brand-red flex items-center justify-center rounded-lg border border-white/10 hover:border-brand-red transition-all duration-300 hover:shadow-lg hover:shadow-brand-red/30">

                            <i class="ri-tiktok-fill text-xl text-gray-400 group-hover:text-white transition-colors"></i>

                        </a>

                    </div>

                </div>



                <!-- Columna 2: Enlaces (20%) -->

                <div class="col-span-1 md:col-span-2">

                    <h4 class="font-display font-bold text-white uppercase text-sm mb-6 tracking-widest relative inline-block">

                        Navegación

                        <div class="absolute -bottom-2 left-0 w-12 h-0.5 bg-brand-red"></div>

                    </h4>

                    <ul class="space-y-3 text-gray-400 font-medium text-sm">

                        <li><a href="#catalogo" class="hover:text-brand-red transition-colors flex items-center gap-2 group">

                            <i class="ri-arrow-right-s-line text-brand-red opacity-0 group-hover:opacity-100 transition-opacity"></i> 

                            <span>Catálogo</span>

                        </a></li>

                        <li><a href="#nosotros" class="hover:text-brand-red transition-colors flex items-center gap-2 group">

                            <i class="ri-arrow-right-s-line text-brand-red opacity-0 group-hover:opacity-100 transition-opacity"></i> 

                            <span>Nosotros</span>

                        </a></li>

                        <li><a href="#guias" class="hover:text-brand-red transition-colors flex items-center gap-2 group">

                            <i class="ri-arrow-right-s-line text-brand-red opacity-0 group-hover:opacity-100 transition-opacity"></i> 

                            <span>Blog</span>

                        </a></li>

                        <li><a href="#contacto" class="hover:text-brand-red transition-colors flex items-center gap-2 group">

                            <i class="ri-arrow-right-s-line text-brand-red opacity-0 group-hover:opacity-100 transition-opacity"></i> 

                            <span>Sedes</span>

                        </a></li>

                    </ul>

                </div>



                <!-- Columna 3: Horarios (20%) -->

                <div class="col-span-1 md:col-span-2">

                    <h4 class="font-display font-bold text-white uppercase text-sm mb-6 tracking-widest relative inline-block">

                        Horarios

                        <div class="absolute -bottom-2 left-0 w-12 h-0.5 bg-brand-yellow"></div>

                    </h4>

                    <div class="space-y-4 text-gray-400 text-sm">

                        <div>

                            <p class="text-white font-semibold mb-1 flex items-center gap-2">

                                <i class="ri-calendar-line text-brand-yellow"></i>

                                Lunes a Viernes

                            </p>

                            <p class="pl-6">9:00 AM - 8:00 PM</p>

                        </div>

                        <div>

                            <p class="text-white font-semibold mb-1 flex items-center gap-2">

                                <i class="ri-calendar-check-line text-brand-yellow"></i>

                                Sábados

                            </p>

                            <p class="pl-6">9:00 AM - 8:00 PM</p>

                        </div>

                        <div>

                            <p class="text-white font-semibold mb-1 flex items-center gap-2">

                                <i class="ri-calendar-check-line text-brand-yellow"></i>

                                Domingos

                            </p>

                            <p class="pl-6 text-gray-500">10:00 AM - 6:00 PM</p>

                        </div>

                    </div>

                </div>



                <!-- Columna 4: Contacto (20%) -->

                <div class="col-span-1 md:col-span-3">

                    <h4 class="font-display font-bold text-white uppercase text-sm mb-6 tracking-widest relative inline-block">

                        Contacto

                        <div class="absolute -bottom-2 left-0 w-12 h-0.5 bg-brand-red"></div>

                    </h4>

                    <ul class="space-y-4 text-gray-400 font-medium text-sm">

                        <li class="flex items-start gap-3 group">

                            <i class="ri-whatsapp-fill text-brand-red text-lg mt-0.5 group-hover:scale-110 transition-transform"></i>

                            <div>

                                <p class="text-white font-semibold text-xs uppercase tracking-wider mb-0.5">WhatsApp</p>

                                <a href="https://wa.me/51982801226" class="hover:text-brand-red transition-colors">+51 982 801 226</a>

                            </div>

                        </li>

                        <li class="flex items-start gap-3 group">

                            <i class="ri-mail-fill text-brand-red text-lg mt-0.5 group-hover:scale-110 transition-transform"></i>

                            <div>

                                <p class="text-white font-semibold text-xs uppercase tracking-wider mb-0.5">Email</p>

                                <a href="mailto:jfdimportacionesperu@gmail.com" class="hover:text-brand-red transition-colors">jfdimportacionesperu@gmail.com</a>

                            </div>

                        </li>

                        <li class="flex items-start gap-3 group">

                            <i class="ri-time-fill text-brand-red text-lg mt-0.5 group-hover:scale-110 transition-transform"></i>

                            <div>

                                <p class="text-white font-semibold text-xs uppercase tracking-wider mb-0.5">Horario</p>

                                <p>Lun-Sáb: 9:00am - 8:00pm</p>

                            </div>

                        </li>

                    </ul>

                </div>

            </div>



            <!-- Separador -->

            <div class="h-px bg-gradient-to-r from-transparent via-white/10 to-transparent mb-8"></div>



            <!-- Bottom bar con crédito de Consultoría Quanta -->

            <div class="flex flex-col md:flex-row items-center justify-between gap-4 text-sm text-gray-500">

                <p class="flex items-center gap-2">

                    © 2026 JFD Importaciones. 

                    <span class="hidden sm:inline">Todos los derechos reservados.</span>

                    <i class="ri-sound-module-fill text-brand-red"></i>

                </p>

            </div>

        </div>

    </footer>



    <div class="fixed bottom-6 right-6 z-[60] flex flex-col items-center gap-4">

    

    <a href="https://wa.me/51989928670?text=Hola%20JFD%20Importaciones%2C%20estoy%20interesado%20en%20sus%20productos." 

       target="_blank" 

       id="whatsapp-button"

       class="group relative flex items-center justify-center transition-all duration-500 hover:-translate-y-1">

        

        <div class="absolute inset-0 rounded-full animate-ping bg-[#25D366] opacity-20"></div>

        <div class="absolute inset-0 rounded-full animate-pulse bg-[#25D366] opacity-40"></div>

        

        <div class="relative w-16 h-16 bg-gradient-to-br from-[#25D366] to-[#128C7E] rounded-full flex items-center justify-center shadow-[0_10px_30px_rgba(37,211,102,0.4)] group-hover:shadow-[0_15px_40px_rgba(37,211,102,0.6)] transition-all overflow-hidden border-2 border-white/20">

            <i class="ri-whatsapp-fill text-4xl text-white transition-transform duration-500 group-hover:scale-110 group-hover:rotate-12"></i>

            

            <div class="absolute top-0 -left-full w-full h-full bg-gradient-to-r from-transparent via-white/20 to-transparent skew-x-[-25deg] group-hover:left-[150%] transition-all duration-1000"></div>

        </div>



        <div class="absolute right-20 top-1/2 -translate-y-1/2 bg-gray-900 text-white px-4 py-2 rounded-lg text-sm font-bold opacity-0 group-hover:opacity-100 translate-x-4 group-hover:translate-x-0 transition-all duration-300 pointer-events-none whitespace-nowrap shadow-xl border border-white/10">

            <span class="flex items-center gap-2">

                <span class="w-2 h-2 bg-green-500 rounded-full animate-pulse"></span>

                ¡Asesoría en línea!

            </span>

            <div class="absolute right-[-6px] top-1/2 -translate-y-1/2 rotate-45 w-3 h-3 bg-gray-900 border-r border-t border-white/10"></div>

        </div>

    </a>



    <button id="back-to-top" 

            class="w-14 h-14 bg-brand-red text-white rounded-full flex items-center justify-center shadow-2xl opacity-0 translate-y-10 pointer-events-none transition-all duration-500 hover:bg-brand-red_hover hover:-translate-y-1 group relative overflow-hidden">

        <div class="absolute inset-0 bg-white/20 translate-y-full group-hover:translate-y-0 transition-transform duration-300"></div>

        <i class="ri-arrow-up-s-line text-3xl relative z-10 group-hover:animate-bounce"></i>

    </button>

</div>



<div id="product-modal" class="fixed inset-0 bg-brand-black/90 z-[100] hidden items-center justify-center p-3 sm:p-4 backdrop-blur-sm opacity-0 transition-all duration-300">
    
    <div id="product-modal-content" class="relative bg-white dark:bg-[#0A0A0A] w-full max-w-5xl max-h-[95vh] md:max-h-[90vh] rounded-xl overflow-hidden flex flex-col md:flex-row shadow-2xl transform scale-95 translate-y-4 transition-all duration-300 border border-gray-200 dark:border-white/10">
        
        <button onclick="closeProductModal()" class="absolute top-3 right-3 sm:top-4 sm:right-4 z-50 w-8 h-8 sm:w-10 sm:h-10 bg-white/80 dark:bg-black/50 hover:bg-brand-red hover:text-white rounded-full flex items-center justify-center text-gray-800 dark:text-gray-300 backdrop-blur transition-all shadow-md">
            <i class="ri-close-line text-xl sm:text-2xl font-bold"></i>
        </button>

        <div class="w-full md:w-1/2 bg-gray-50 dark:bg-[#111] relative flex items-center justify-center p-4 sm:p-8 border-b md:border-b-0 md:border-r border-gray-200 dark:border-white/5 flex-shrink-0 h-[35vh] sm:h-[40vh] md:h-auto">
            <div class="absolute inset-0 flex items-center justify-center opacity-[0.03] select-none pointer-events-none overflow-hidden">
                <span id="modal-brand-bg" class="font-display font-black text-5xl sm:text-7xl md:text-8xl uppercase tracking-tighter text-black dark:text-white whitespace-nowrap"></span>
            </div>
            <img id="modal-image" src="" class="relative z-10 w-full h-full object-contain drop-shadow-2xl transition-transform duration-500 hover:scale-105" alt="Producto">
        </div>
        
        <div class="w-full md:w-1/2 p-5 sm:p-6 md:p-10 flex flex-col justify-between overflow-y-auto custom-scrollbar bg-white dark:bg-[#0A0A0A] flex-1">
            
            <div>
                <div class="flex items-center gap-3 mb-3 mt-2 md:mt-0">
                    <span id="modal-brand" class="bg-brand-red/10 text-brand-red px-3 py-1 rounded text-xs font-black uppercase tracking-widest"></span>
                    <span id="modal-category" class="text-[10px] font-bold text-gray-500 uppercase tracking-wider line-clamp-1"></span>
                </div>
                
                <h3 id="modal-title" class="font-display font-black text-xl sm:text-2xl md:text-3xl text-gray-900 dark:text-white leading-tight uppercase mb-4"></h3>
                
                <p id="modal-description" class="text-sm text-gray-600 dark:text-gray-400 leading-relaxed mb-6"></p>

                <div class="bg-gray-50 dark:bg-white/5 rounded-lg p-4 mb-6 border border-gray-100 dark:border-white/5">
                    <h4 class="text-[10px] font-black uppercase tracking-widest text-brand-red mb-3 flex items-center gap-2">
                        <i class="ri-equalizer-line"></i> Especificaciones
                    </h4>
                    <div id="modal-specs-container" class="grid grid-cols-1 sm:grid-cols-2 gap-3">
                    </div>
                </div>
            </div>

            <div class="mt-2 pt-4 sm:pt-6 border-t border-gray-100 dark:border-white/10 flex-shrink-0">
                <div class="flex items-end justify-between mb-4 sm:mb-6">
                    <div>
                        <p class="text-[10px] text-gray-500 uppercase tracking-widest mb-1 font-bold">Precio Unitario</p>
                        <div class="flex items-baseline gap-2">
                            <span class="text-xs sm:text-sm font-bold text-brand-red uppercase">S/</span>
                            <span class="text-3xl sm:text-4xl font-display font-black text-gray-900 dark:text-white tracking-tighter" id="modal-price"></span>
                        </div>
                    </div>
                </div>

                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3">
                    <button id="modal-add-cart" class="w-full bg-brand-red hover:bg-brand-red_hover text-white font-bold text-sm uppercase tracking-widest py-3 sm:py-4 rounded transition-all shadow-lg hover:shadow-brand-red/30 flex items-center justify-center gap-2 active:scale-95">
                        <i class="ri-shopping-cart-2-line text-lg"></i> Agregar
                    </button>
                    <a id="modal-whatsapp-link" href="#" target="_blank" class="w-full bg-brand-yellow hover:bg-yellow-400 text-brand-black font-bold text-sm uppercase tracking-widest py-3 sm:py-4 rounded transition-all shadow-lg flex items-center justify-center gap-2 active:scale-95">
                        <i class="ri-whatsapp-line text-lg"></i> Consultar
                    </a>
                </div>
            </div>

        </div>
    </div>
</div>



<div id="article-modal" class="fixed inset-0 bg-black/95 z-[100] hidden items-center justify-center p-0 md:p-6 backdrop-blur-xl opacity-0 transition-all duration-500">

    <div id="article-modal-content" class="relative bg-[#0A0A0A] w-full max-w-5xl h-full md:h-[90vh] overflow-hidden flex flex-col shadow-2xl transform translate-y-10 transition-all duration-500 border border-white/10 text-white">

        

        <button onclick="closeArticleModal()" class="absolute top-6 right-6 z-50 bg-black/50 text-white hover:bg-brand-red rounded-full p-2 transition-colors shadow-lg backdrop-blur-md border border-white/10">

            <i class="ri-close-line text-2xl font-bold"></i>

        </button>



        <div class="relative h-[40vh] md:h-[50vh] flex-shrink-0 overflow-hidden bg-black">

            <img id="article-image" src="" class="absolute inset-0 w-full h-full object-cover opacity-60 transition-transform duration-1000" alt="">

            <div class="absolute inset-0 bg-gradient-to-t from-[#0A0A0A] via-transparent to-transparent"></div>

            

            <div class="absolute bottom-0 left-0 w-full p-8 md:p-12">

                <div class="flex items-center gap-4 mb-4">

                    <span id="article-category" class="bg-brand-red text-white text-[10px] font-black uppercase tracking-[0.3em] px-4 py-1 shadow-lg"></span>

                </div>

                <h2 id="article-title" class="font-display font-bold text-4xl md:text-6xl text-white leading-[0.9] tracking-tighter uppercase max-w-3xl drop-shadow-lg"></h2>

            </div>

        </div>



        <div class="flex-1 overflow-y-auto bg-[#0A0A0A] p-8 md:p-16 custom-scrollbar">

            <div class="max-w-3xl mx-auto">

                <div class="flex items-center gap-6 mb-12 text-[10px] font-bold uppercase tracking-widest text-gray-400 border-b border-white/10 pb-6">

                    <span class="flex items-center gap-2"><i class="ri-calendar-event-line text-brand-red"></i> <span id="article-date"></span></span>

                    <span class="flex items-center gap-2"><i class="ri-time-line text-brand-red"></i> <span id="article-read-time"></span></span>

                    <span class="flex items-center gap-2"><i class="ri-shield-user-line text-brand-red"></i> Staff JFD.PRO</span>

                </div>



                <div id="article-content" class="prose prose-lg prose-invert max-w-none 

                    prose-headings:font-display prose-headings:uppercase prose-headings:tracking-tighter

                    prose-strong:text-brand-red prose-strong:font-bold

                    prose-p:text-gray-300 prose-p:font-light prose-p:leading-relaxed">

                </div>



                <div class="mt-20 pt-10 border-t border-white/10 flex flex-col md:flex-row justify-between items-center gap-8">

                    <div class="text-center md:text-left">

                        <p class="text-xs text-gray-500 uppercase tracking-widest mb-2 font-bold">¿Dudas con este tema?</p>

                        <p class="text-white font-display font-bold text-xl">Habla con un especialista hoy.</p>

                    </div>

                    <div class="flex gap-4 w-full md:w-auto">

                        <a href="https://wa.me/51989928670" target="_blank" class="flex-1 md:flex-none bg-[#25D366] text-white px-8 py-4 font-bold uppercase tracking-widest text-sm hover:bg-[#128C7E] transition-all flex items-center justify-center gap-3 rounded shadow-lg">

                            <i class="ri-whatsapp-line text-xl"></i> WhatsApp

                        </a>

                    </div>

                </div>

            </div>

        </div>

    </div>

</div>

<div id="cart-overlay" onclick="toggleCart()" class="fixed inset-0 bg-black/80 z-[60] hidden transition-opacity opacity-0 backdrop-blur-sm"></div>





    <div id="cart-overlay" onclick="toggleCart()" class="fixed inset-0 bg-black/80 z-[60] hidden transition-opacity opacity-0 backdrop-blur-sm"></div>



    <div id="cart-sidebar" class="fixed inset-y-0 right-0 w-full md:w-[450px] bg-brand-light dark:bg-brand-dark z-[70] transform translate-x-full transition-transform duration-300 shadow-2xl flex flex-col border-l border-gray-200 dark:border-white/5">

        <div class="h-24 bg-brand-black flex items-center justify-between px-8 border-b-2 border-brand-red relative overflow-hidden">

            <div class="absolute inset-0 bg-[url('https://www.transparenttextures.com/patterns/carbon-fibre.png')] opacity-20 mix-blend-overlay"></div>

            <div class="flex items-center gap-3 text-white relative z-10">

                <i class="ri-shopping-cart-2-fill text-brand-red text-3xl"></i>

                <h3 class="font-display font-black text-3xl uppercase tracking-wide">Tu Pedido</h3>

            </div>

            <button onclick="toggleCart()" class="text-gray-400 hover:text-white transition-colors relative z-10 bg-white/10 p-2 rounded-full hover:bg-brand-red">

                <i class="ri-close-line text-3xl"></i>

            </button>

        </div>



        <div id="cart-items" class="flex-1 overflow-y-auto p-6 space-y-4 bg-gray-100 dark:bg-brand-dark/50"></div>



        <div class="p-8 bg-white dark:bg-brand-black border-t border-gray-200 dark:border-white/10 shadow-[0_-10px_30px_rgba(0,0,0,0.2)] relative z-20">

            <div class="flex justify-between items-end mb-6">

                <span class="text-base font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest">Total Estimado</span>

                <span id="cart-total" class="text-4xl font-display font-black text-brand-red">S/ 0.00</span>

            </div>

            <button onclick="checkout()" class="group w-full py-5 bg-brand-yellow hover:bg-white text-brand-black font-black font-display text-2xl uppercase tracking-wider rounded-sm shadow-neon transition-all flex items-center justify-center gap-3 relative overflow-hidden">

                <span class="relative z-10 flex items-center gap-2"><i class="ri-whatsapp-line text-3xl"></i> Cotizar Ahora</span>

                <div class="absolute inset-0 w-full h-full bg-white/20 scale-x-0 group-hover:scale-x-100 transition-transform origin-left"></div>

            </button>

            <p class="text-center text-gray-500 text-xs mt-4 flex items-center justify-center gap-2">

                <i class="ri-shield-check-line text-brand-yellow"></i> Compra segura. Te atenderá un ingeniero de JFD.

            </p>

        </div>

    </div>



<script>

// ============================================================

// 1. DATA E INICIALIZACIÓN

// ============================================================

// ============================================================

// 1. DATA E INICIALIZACIÓN

// ============================================================



// ¡ESTA ES LA LÍNEA QUE FALTABA! Sin esto, los filtros rompen la página:

const categoriesTree = <?php echo json_encode($categoriesJS); ?>; 



const allProducts = <?php echo json_encode($productsJS); ?>;

const categoriesList = <?php echo json_encode($categories); ?>; 



// Datos del Blog

// --- DATOS DEL BLOG (ACTUALIZADO: 5 NOTICIAS) ---

// --- DATOS DEL BLOG (ACTUALIZADO: IMAGEN DE CABLES MEJORADA) ---

// --- DATOS DEL BLOG (CONTENIDO ENRIQUECIDO E INTERACTIVO) ---
// --- DATOS DEL BLOG (NIVEL PRO: CAJAS, GRIDS Y ALERTAS) ---
const blogPosts = {
    'potencia-espacio': {
        title: "¿QUÉ POTENCIA NECESITO?",
        date: "13 FEB 2026",
        readTime: "5 MIN",
        image: "https://images.unsplash.com/photo-1574169208507-84376144848b?q=80&w=800",
        category: "GUÍA TÉCNICA",
        content: `
            <p style="font-size: 1.15rem; color: #aaa; margin-bottom: 2rem; line-height: 1.6;">
                Un error común es comprar "el parlante más grande" para una sala pequeña, o quedarse corto en un evento al aire libre. La acústica es una ciencia, no una adivinanza.
            </p>
            
            <div style="background: linear-gradient(145deg, rgba(217,4,41,0.1) 0%, rgba(0,0,0,0) 100%); border-left: 4px solid #D90429; padding: 1.5rem; border-radius: 0 8px 8px 0; margin-bottom: 2rem;">
                <h4 style="color: #D90429; margin: 0 0 0.5rem 0; font-weight: bold; text-transform: uppercase; letter-spacing: 1px;">ALERTA: ¿PMPO o RMS?</h4>
                <p style="margin: 0; color: #ddd; font-size: 0.95rem;">
                    Ignora cualquier etiqueta que diga "50,000W PMPO". Ese número es puro marketing inflado. Fíjate SIEMPRE en los <strong>Watts RMS</strong>, que es la potencia real, continua y medible que el equipo puede soportar sin quemarse.
                </p>
            </div>
            
            <h3 style="color: white; margin: 2.5rem 0 1.5rem 0; border-bottom: 1px solid rgba(255,255,255,0.1); padding-bottom: 0.5rem;">La Regla de Oro: Cálculo por Persona</h3>
            <p style="margin-bottom: 1.5rem;">Para interiores, calcula <strong>5 a 10 watts RMS por persona</strong>. Para exteriores, necesitas al menos el doble (10 a 20 watts) porque no hay paredes que contengan y reboten el sonido.</p>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(240px, 1fr)); gap: 1.5rem; margin-bottom: 2.5rem;">
                <div style="background: rgba(255,255,255,0.03); border: 1px solid rgba(255,255,255,0.05); padding: 1.5rem; border-radius: 8px; text-align: center;">
                    <div style="width: 50px; height: 50px; background: rgba(255,255,255,0.1); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem auto;">
                        <span style="font-size: 1.5rem;">🏠</span>
                    </div>
                    <h5 style="color: white; margin: 0 0 0.5rem 0; font-size: 1.1rem;">Sala / Reunión</h5>
                    <p style="color: #888; font-size: 0.85rem; margin-bottom: 1rem; text-transform: uppercase;">Hasta 50 personas</p>
                    <div style="color: #FFD60A; font-weight: bold; font-size: 1.25rem;">300W - 500W RMS</div>
                </div>

                <div style="background: rgba(255,255,255,0.03); border: 1px solid rgba(255,255,255,0.05); padding: 1.5rem; border-radius: 8px; text-align: center;">
                    <div style="width: 50px; height: 50px; background: rgba(255,255,255,0.1); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem auto;">
                        <span style="font-size: 1.5rem;">🏢</span>
                    </div>
                    <h5 style="color: white; margin: 0 0 0.5rem 0; font-size: 1.1rem;">Fiesta Local Cerrado</h5>
                    <p style="color: #888; font-size: 0.85rem; margin-bottom: 1rem; text-transform: uppercase;">100 - 200 personas</p>
                    <div style="color: #FFD60A; font-weight: bold; font-size: 1.25rem;">1000W - 2000W RMS</div>
                </div>

                <div style="background: rgba(217,4,41,0.05); border: 1px solid rgba(217,4,41,0.2); padding: 1.5rem; border-radius: 8px; text-align: center;">
                    <div style="width: 50px; height: 50px; background: rgba(217,4,41,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem auto;">
                        <span style="font-size: 1.5rem;">🎪</span>
                    </div>
                    <h5 style="color: white; margin: 0 0 0.5rem 0; font-size: 1.1rem;">Exterior / Concierto</h5>
                    <p style="color: #888; font-size: 0.85rem; margin-bottom: 1rem; text-transform: uppercase;">300+ personas</p>
                    <div style="color: #D90429; font-weight: bold; font-size: 1.25rem;">4000W+ RMS (Line Array)</div>
                </div>
            </div>

            <div style="background: #111; padding: 1.5rem; border-radius: 8px; border: 1px dashed rgba(255,214,10,0.5);">
                <h4 style="color: #FFD60A; margin: 0 0 0.5rem 0;">💡 Pro Tip JFD: El Headroom</h4>
                <p style="margin: 0; color: #ccc; font-size: 0.95rem;">
                    Nunca trabajes tu equipo al 100% de su capacidad. Si necesitas 1000W para tu evento, compra un sistema de 1500W. Ese margen del 20-30% se llama <em>Headroom</em> y garantiza que el sonido sea limpio, sin distorsión, y que tus equipos te duren años.
                </p>
            </div>
        `
    },
    'primer-equipo': {
        title: "TU PRIMER EQUIPO DE SONIDO",
        date: "12 FEB 2026",
        readTime: "7 MIN",
        image: "https://images.unsplash.com/photo-1511671782779-c97d3d27a1d4?q=80&w=800",
        category: "COMPRA INTELIGENTE",
        content: `
            <p style="font-size: 1.15rem; color: #aaa; margin-bottom: 2rem; line-height: 1.6;">
                No gastes tu presupuesto en luces si tu sonido es malo. El éxito de un evento depende en un 80% de la claridad del audio. Aquí te decimos cómo invertir tu dinero estratégicamente.
            </p>
            
            <h3 style="color: white; margin: 2rem 0 1rem 0;">La Cadena de Audio: ¿Dónde pongo el dinero?</h3>
            <p style="margin-bottom: 1.5rem;">La cadena de audio es tan fuerte como su eslabón más débil. De nada sirve un sistema de parlantes de S/ 5000 si usas descargas MP3 de mala calidad.</p>

            <div style="display: flex; flex-direction: column; gap: 1rem; margin-bottom: 2.5rem;">
                <div style="display: flex; align-items: center; background: rgba(255,255,255,0.03); padding: 1rem; border-radius: 8px; border-left: 3px solid #666;">
                    <div style="min-width: 40px; font-weight: bold; color: #666;">01</div>
                    <div>
                        <h5 style="color: white; margin: 0 0 0.2rem 0;">La Fuente</h5>
                        <p style="margin: 0; color: #888; font-size: 0.9rem;">Archivos WAV, FLAC o MP3 a 320kbps. Spotify Premium configurado en calidad "Muy Alta".</p>
                    </div>
                </div>
                <div style="display: flex; align-items: center; background: rgba(255,255,255,0.03); padding: 1rem; border-radius: 8px; border-left: 3px solid #FFD60A;">
                    <div style="min-width: 40px; font-weight: bold; color: #FFD60A;">02</div>
                    <div>
                        <h5 style="color: white; margin: 0 0 0.2rem 0;">La Consola (Mezcladora)</h5>
                        <p style="margin: 0; color: #888; font-size: 0.9rem;">Invierte en una consola pequeña pero de marca confiable (Soundcraft, Yamaha, Mackie) por sus preamplificadores limpios.</p>
                    </div>
                </div>
                <div style="display: flex; align-items: center; background: rgba(255,255,255,0.03); padding: 1rem; border-radius: 8px; border-left: 3px solid #D90429;">
                    <div style="min-width: 40px; font-weight: bold; color: #D90429;">03</div>
                    <div>
                        <h5 style="color: white; margin: 0 0 0.2rem 0;">Parlantes Activos</h5>
                        <p style="margin: 0; color: #888; font-size: 0.9rem;">El destino final. Busca marcas con buen soporte técnico local y repuestos garantizados.</p>
                    </div>
                </div>
            </div>

            <h3 style="color: white; margin: 2rem 0 1.5rem 0;">El Debate Eterno: ¿Activo o Pasivo?</h3>
            <p>Para tu primer equipo, la respuesta casi siempre es <strong>ACTIVO</strong>. Aquí el por qué:</p>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1.5rem; margin-top: 1.5rem;">
                <div style="background: rgba(37,211,102,0.05); border: 1px solid rgba(37,211,102,0.2); padding: 1.5rem; border-radius: 8px;">
                    <h4 style="color: #25D366; margin: 0 0 1rem 0; font-size: 1.1rem; text-align: center;">✅ Ventajas del Activo</h4>
                    <ul style="margin: 0; padding-left: 1.2rem; color: #ddd; font-size: 0.9rem; line-height: 1.6;">
                        <li>Amplificador 100% calibrado de fábrica para ese parlante específico.</li>
                        <li>Procesadores (DSP) integrados que evitan que lo quemes.</li>
                        <li>Menos peso, menos cables, menos cajas que cargar.</li>
                        <li>Plug and play: conectas a la corriente, mandas señal y listo.</li>
                    </ul>
                </div>
                <div style="background: rgba(217,4,41,0.05); border: 1px solid rgba(217,4,41,0.2); padding: 1.5rem; border-radius: 8px;">
                    <h4 style="color: #D90429; margin: 0 0 1rem 0; font-size: 1.1rem; text-align: center;">❌ Lo difícil del Pasivo</h4>
                    <ul style="margin: 0; padding-left: 1.2rem; color: #ddd; font-size: 0.9rem; line-height: 1.6;">
                        <li>Tienes que saber hacer "match" de impedancias (Ohmios) y potencias.</li>
                        <li>Requiere racks de amplificación pesados externos.</li>
                        <li>Necesitas crossovers externos para separar bajos de agudos.</li>
                        <li>Si te equivocas en la configuración, quemas el equipo en segundos.</li>
                    </ul>
                </div>
            </div>
        `
    },
    'casa-vs-eventos': {
        title: "CASA VS. EVENTOS: LA VERDAD",
        date: "10 FEB 2026",
        readTime: "6 MIN",
        image: "https://images.unsplash.com/photo-1492684223066-81342ee5ff30?q=80&w=800",
        category: "COMPARATIVA",
        content: `
            <p style="font-size: 1.15rem; color: #aaa; margin-bottom: 2rem; line-height: 1.6;">
                Cada fin de semana, recibimos clientes que quemaron sus costosos equipos "de casa" (Minicomponentes, barras de sonido grandes) intentando sonorizar una fiesta de 50 personas. Entiende las diferencias de ADN para no perder tu inversión.
            </p>
            
            <div style="background: #111; padding: 2rem; border-radius: 8px; margin-bottom: 2.5rem; text-align: center; border: 1px solid rgba(255,255,255,0.05);">
                <h3 style="color: white; margin-top: 0;">Fidelidad vs. Resistencia Extrema</h3>
                <p style="color: #888; margin-bottom: 0;">Un equipo casero (Hi-Fi) está diseñado para sonar hermoso a volúmenes bajos o medios en un entorno cerrado. Un equipo Profesional está construido como un tanque para sobrevivir a la guerra del directo.</p>
            </div>

            <div style="display: flex; flex-wrap: wrap; gap: 1.5rem; margin-bottom: 2.5rem;">
                <div style="flex: 1; min-width: 280px; background: rgba(255,255,255,0.02); padding: 1.5rem; border-radius: 8px; border-top: 4px solid #666;">
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 1rem;">
                        <i class="ri-home-smile-line" style="font-size: 1.5rem; color: #666;"></i>
                        <h4 style="color: white; margin: 0; font-size: 1.2rem;">Equipo Casero (Hi-Fi)</h4>
                    </div>
                    <ul style="margin: 0; padding-left: 1.2rem; color: #bbb; font-size: 0.95rem; line-height: 1.7;">
                        <li>Componentes delicados (conos de cartón/papel).</li>
                        <li>Disipación de calor <strong>pasiva</strong> (sin ventiladores).</li>
                        <li>Bobinas pequeñas que se saturan rápido.</li>
                        <li><strong style="color: #D90429;">Fallo común:</strong> Si lo pones al 100% de volumen por 2 horas en una fiesta, el calor derrite la bobina.</li>
                    </ul>
                </div>

                <div style="flex: 1; min-width: 280px; background: rgba(217,4,41,0.03); padding: 1.5rem; border-radius: 8px; border-top: 4px solid #D90429;">
                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 1rem;">
                        <i class="ri-speaker-fill" style="font-size: 1.5rem; color: #D90429;"></i>
                        <h4 style="color: white; margin: 0; font-size: 1.2rem;">Equipo PRO Audio</h4>
                    </div>
                    <ul style="margin: 0; padding-left: 1.2rem; color: #bbb; font-size: 0.95rem; line-height: 1.7;">
                        <li>Conos reforzados con kevlar e imanes de neodimio/ferrita masivos.</li>
                        <li>Disipadores gigantes y ventiladores <strong>activos</strong>.</li>
                        <li>Protecciones DSP contra picos de voltaje.</li>
                        <li><strong style="color: #25D366;">Diseñados para:</strong> Trabajar al 90% de su capacidad durante 8 a 12 horas seguidas sin inmutarse.</li>
                    </ul>
                </div>
            </div>

            <div style="background: rgba(255,214,10,0.1); border-left: 4px solid #FFD60A; padding: 1.5rem; border-radius: 0 8px 8px 0;">
                <h4 style="color: #FFD60A; margin: 0 0 0.5rem 0; text-transform: uppercase;">El Veredicto de JFD</h4>
                <p style="margin: 0; color: #ddd;">
                    Si cobras por el evento, si hay alcohol de por medio, o si simplemente quieres hacer temblar los vidrios en un patio grande sin el estrés de quemar nada... <strong>necesitas equipo PRO</strong>. Deja el equipo de casa para ver Netflix tranquilo en tu sofá.
                </p>
            </div>
        `
    },
    'guia-conexiones': {
        title: "GUÍA RÁPIDA DE CONEXIONES",
        date: "08 FEB 2026",
        readTime: "4 MIN",
        image: "https://png.pngtree.com/thumb_back/fw800/background/20251210/pngtree-close-up-of-a-modular-synthesizer-with-colorful-patch-cables-connecting-image_20786178.webp",
        category: "PRINCIPIANTES",
        content: `
            <p style="font-size: 1.15rem; color: #aaa; margin-bottom: 2rem; line-height: 1.6;">
                El 90% de los ruidos extraños, zumbidos "de tierra" y fallos en pleno evento se deben a usar el cable equivocado. Aquí tu mapa de supervivencia en cablería.
            </p>
            
            <div style="background: rgba(255,255,255,0.02); border: 1px solid rgba(255,255,255,0.05); padding: 2rem; border-radius: 8px; margin-bottom: 2.5rem;">
                <h3 style="color: white; margin-top: 0; text-align: center;">La Regla de los 5 Metros</h3>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 2rem; margin-top: 1.5rem;">
                    <div>
                        <h4 style="color: #D90429; margin-bottom: 0.5rem;">Cable Desbalanceado</h4>
                        <p style="color: #888; font-size: 0.9rem;">Solo lleva 2 hilos (Señal y Tierra). <strong>Problema:</strong> Si el cable mide más de 5 metros, actúa como una antena y capta interferencias de luces, radios o celulares.</p>
                    </div>
                    <div>
                        <h4 style="color: #25D366; margin-bottom: 0.5rem;">Cable Balanceado</h4>
                        <p style="color: #888; font-size: 0.9rem;">Lleva 3 hilos (Tierra, Señal Positiva y Señal Negativa invertida). Al llegar al final, se vuelve a invertir, <strong>cancelando mágicamente todo el ruido externo</strong>. Puedes tirar 100 metros de cable sin ruido.</p>
                    </div>
                </div>
            </div>

            <h3 style="color: white; margin-bottom: 1.5rem;">El Arsenal Analógico: Los 4 Fantásticos</h3>
            
            <div style="display: flex; flex-direction: column; gap: 1rem;">
                <div style="background: #111; padding: 1.5rem; border-radius: 8px; display: flex; align-items: center; gap: 1.5rem; border-left: 3px solid #FFD60A;">
                    <div style="background: #222; width: 60px; height: 60px; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-weight: black; color: white;">XLR</div>
                    <div>
                        <h4 style="color: white; margin: 0 0 0.3rem 0;">El Canon (Balanceado)</h4>
                        <p style="color: #888; margin: 0; font-size: 0.9rem;">El estándar de la industria. Tiene 3 pines y un seguro contra tirones. Úsalo SIEMPRE que puedas para conectar micrófonos, consolas y parlantes activos.</p>
                    </div>
                </div>

                <div style="background: #111; padding: 1.5rem; border-radius: 8px; display: flex; align-items: center; gap: 1.5rem; border-left: 3px solid #FFD60A;">
                    <div style="background: #222; width: 60px; height: 60px; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-weight: black; color: white;">TRS</div>
                    <div>
                        <h4 style="color: white; margin: 0 0 0.3rem 0;">Jack 1/4" Estéreo (Balanceado)</h4>
                        <p style="color: #888; margin: 0; font-size: 0.9rem;">Físicamente tiene "dos rayitas" negras en la punta. Hace exactamente la misma función de cancelación de ruido que el XLR, pero en formato plug.</p>
                    </div>
                </div>

                <div style="background: #111; padding: 1.5rem; border-radius: 8px; display: flex; align-items: center; gap: 1.5rem; border-left: 3px solid #666;">
                    <div style="background: #222; width: 60px; height: 60px; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-weight: black; color: white;">TS</div>
                    <div>
                        <h4 style="color: white; margin: 0 0 0.3rem 0;">Jack 1/4" Mono (Desbalanceado)</h4>
                        <p style="color: #888; margin: 0; font-size: 0.9rem;">Una sola rayita en la punta. Solo sirve para tramos cortos, como conectar una guitarra o bajo directamente a su amplificador o pedalera.</p>
                    </div>
                </div>

                <div style="background: #111; padding: 1.5rem; border-radius: 8px; display: flex; align-items: center; gap: 1.5rem; border-left: 3px solid #3B82F6;">
                    <div style="background: #222; width: 60px; height: 60px; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-weight: black; color: white;">SPK</div>
                    <div>
                        <h4 style="color: white; margin: 0 0 0.3rem 0;">Speakon (Potencia)</h4>
                        <p style="color: #888; margin: 0; font-size: 0.9rem;">Conector robusto con giro de bloqueo de seguridad. <strong>Ojo:</strong> No lleva señal, lleva ENERGÍA de alta corriente desde un rack amplificador hacia un parlante pasivo.</p>
                    </div>
                </div>
            </div>
        `
    },
    'cuidado-parlantes': {
        title: "CÓMO CUIDAR TUS PARLANTES",
        date: "05 FEB 2026",
        readTime: "5 MIN",
        image: "https://images.unsplash.com/photo-1520523839897-bd0b52f945a0?q=80&w=800",
        category: "MANTENIMIENTO",
        content: `
            <p style="font-size: 1.15rem; color: #aaa; margin-bottom: 2rem; line-height: 1.6;">
                Un buen parlante profesional es una inversión fuerte. La diferencia entre que dure 10 años o 10 días radica netamente en las manos del operador. Sigue estos 3 mandamientos para no llorar después.
            </p>
            
            <h3 style="color: white; margin: 2.5rem 0 1.5rem 0;">1. La Sagrada Secuencia de Encendido</h3>
            <p style="margin-bottom: 1.5rem;">El error de novato más destructivo es encender o apagar los equipos al revés. Ese fuerte "¡POP!" que truena en tus parlantes al prender la consola puede literalmente arrancar el driver de agudos.</p>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 2.5rem;">
                <div style="background: rgba(37,211,102,0.05); border: 1px solid rgba(37,211,102,0.3); padding: 1.5rem; border-radius: 8px;">
                    <h4 style="color: #25D366; margin: 0 0 1rem 0; text-align: center; text-transform: uppercase; font-size: 1rem;">🟢 Para Encender</h4>
                    <ol style="margin: 0; padding-left: 1.2rem; color: #ddd; font-size: 0.95rem; line-height: 1.8;">
                        <li>Conecta laptops y micrófonos.</li>
                        <li>Enciende la Consola/Mezcladora.</li>
                        <li><strong>Por último:</strong> Enciende los parlantes.</li>
                    </ol>
                </div>
                <div style="background: rgba(217,4,41,0.05); border: 1px solid rgba(217,4,41,0.3); padding: 1.5rem; border-radius: 8px;">
                    <h4 style="color: #D90429; margin: 0 0 1rem 0; text-align: center; text-transform: uppercase; font-size: 1rem;">🔴 Para Apagar</h4>
                    <ol style="margin: 0; padding-left: 1.2rem; color: #ddd; font-size: 0.95rem; line-height: 1.8;">
                        <li><strong>Primero SIEMPRE:</strong> Apaga los parlantes.</li>
                        <li>Apaga la Consola/Mezcladora.</li>
                        <li>Desconecta laptops y periféricos.</li>
                    </ol>
                </div>
            </div>

            <div style="background: linear-gradient(145deg, rgba(217,4,41,0.1) 0%, rgba(0,0,0,0) 100%); border-left: 4px solid #D90429; padding: 1.5rem; border-radius: 0 8px 8px 0; margin-bottom: 2.5rem;">
                <h3 style="color: #D90429; margin: 0 0 0.5rem 0; font-weight: bold; text-transform: uppercase;">2. El Demonio Rojo: El "Clipping"</h3>
                <p style="margin: 0; color: #ddd; font-size: 0.95rem; line-height: 1.6;">
                    ¿Ves ese foquito rojo en tu consola o detrás de tu parlante activo que dice "PEAK", "LIMIT" o "CLIP"? Es un aviso de muerte térmica. Cuando la señal "clipea" (se vuelve cuadrada), el parlante recibe energía distorsionada que sobrecalienta la bobina hasta carbonizarla. <strong>Si parpadea rojo constantemente, bájale al máster inmediatamente. No es negociable.</strong>
                </p>
            </div>

            <h3 style="color: white; margin-bottom: 1.5rem;">3. Climas Extremos: Humedad y Polvo</h3>
            <div style="background: rgba(255,255,255,0.02); border: 1px solid rgba(255,255,255,0.05); padding: 1.5rem; border-radius: 8px;">
                <p style="color: #bbb; margin-top: 0;">Si trabajas en exteriores, en la selva o en la sierra con alta humedad, aplica este checklist:</p>
                <ul style="margin-bottom: 0; padding-left: 1.2rem; color: #ddd; font-size: 0.95rem; line-height: 1.7;">
                    <li><strong>Aclimatación:</strong> Si el equipo estuvo en un camión frío y entra a un local caluroso, espera 30 minutos antes de prenderlo. La condensación interna crea cortocircuitos.</li>
                    <li><strong>Fundas Acolchadas:</strong> No solo evitan rayones, evitan que el polvo bloquee los ventiladores internos del módulo de amplificación.</li>
                    <li><strong>Sílica Gel:</strong> Mete bolsitas de sílica en tus racks o cases. El óxido en la bobina magnética es muerte súbita para un driver.</li>
                </ul>
            </div>
        `
    }
};


// Variables Globales del Catálogo

const ITEMS_PER_PAGE = 9;

let currentPage = 1;

let filteredProducts = [...allProducts]; // Copia para filtrar

let cart = JSON.parse(localStorage.getItem('jfd_cart_vfinal')) || [];



// ============================================================

// 2. LÓGICA DEL CATÁLOGO (FILTROS Y PAGINACIÓN)

// ============================================================



// A. Generar Filtros de Categoría Automáticamente (ESTO FALTABA)

// A. Generar Filtros de Categoría (AGRUPADOS POR FAMILIA)

function renderCategoryFilters() {

    const container = document.getElementById('category-filters');

    if (!container) return;



    container.innerHTML = ''; // Limpiamos lo anterior



    // Recorremos las categorías principales (Audio, Instrumentos, etc.)

    Object.keys(categoriesTree).forEach(mainCat => {

        // 1. Crear el Título de la Categoría Principal

        const groupDiv = document.createElement('div');

        groupDiv.className = 'mb-6 border-b border-gray-200 dark:border-white/5 pb-4 last:border-0';



        const title = document.createElement('h4');

        title.className = 'font-display font-bold text-sm text-gray-900 dark:text-white uppercase mb-3 flex items-center justify-between cursor-pointer hover:text-brand-red transition-colors';

        title.innerHTML = `

            ${mainCat} 

            <i class="ri-arrow-down-s-line text-gray-400"></i>

        `;

        

        // Al hacer click en el título, colapsa o expande la lista (Opcional, pero queda pro)

        title.onclick = () => {

            const list = document.getElementById(`list-${mainCat.replace(/\s/g, '')}`);

            list.classList.toggle('hidden');

        };



        // 2. Crear la lista de Subcategorías (Checkboxes)

        const subList = document.createElement('ul');

        subList.id = `list-${mainCat.replace(/\s/g, '')}`; // ID único sin espacios

        subList.className = 'space-y-2 pl-1';



        const subCats = categoriesTree[mainCat];

        

        subCats.forEach(sub => {

            if (sub === 'Todo') return; // Saltamos la opción "Todo"



            const li = document.createElement('li');

            li.innerHTML = `

                <label class="flex items-center gap-3 cursor-pointer group">

                    <input type="checkbox" value="${sub}" class="filter-category w-4 h-4 accent-brand-red bg-gray-200 border-gray-300 rounded cursor-pointer transition-all" onchange="applyFilters()">

                    <span class="text-xs font-medium text-gray-500 dark:text-gray-400 group-hover:text-brand-red transition-colors uppercase tracking-wide">${sub}</span>

                </label>

            `;

            subList.appendChild(li);

        });



        groupDiv.appendChild(title);

        groupDiv.appendChild(subList);

        container.appendChild(groupDiv);

    });

}



// B. Aplicar Filtros (Buscador, Precio, Categoría)

// B. Aplicar Filtros (ACTUALIZADO PARA SUBCATEGORÍAS)

function applyFilters() {

    const searchTerm = document.getElementById('search-input').value.toLowerCase();

    const minPrice = parseFloat(document.getElementById('min-price').value) || 0;

    const maxPrice = parseFloat(document.getElementById('max-price').value) || Infinity;

    const sortSelect = document.getElementById('sort-select');

    const sortValue = sortSelect ? sortSelect.value : 'newest';

    

    // Obtener qué checkboxes están marcados

    const checkedSubs = Array.from(document.querySelectorAll('.filter-category:checked')).map(cb => cb.value);



    // Filtrar

    filteredProducts = allProducts.filter(product => {

        // 1. Texto

        const matchesSearch = product.name.toLowerCase().includes(searchTerm) || 

                              product.brand.toLowerCase().includes(searchTerm) ||

                              product.sub_cat.toLowerCase().includes(searchTerm);

        

        // 2. Precio

        const matchesPrice = parseFloat(product.price) >= minPrice && parseFloat(product.price) <= maxPrice;

        

        // 3. Categoría (AQUÍ ESTÁ EL CAMBIO CLAVE)

        // Si no hay nada marcado, muestra todo. Si hay marcado, busca coincidencia en SUB_CAT o en CATEGORY

        const matchesCategory = checkedSubs.length === 0 || 

                                checkedSubs.includes(product.sub_cat) || 

                                checkedSubs.includes(product.category);



        return matchesSearch && matchesPrice && matchesCategory;

    });



    // Ordenar (Igual que antes)

    if (sortValue === 'price-asc') {

        filteredProducts.sort((a, b) => a.price - b.price);

    } else if (sortValue === 'price-desc') {

        filteredProducts.sort((a, b) => b.price - a.price);

    } else if (sortValue === 'name-asc') {

        filteredProducts.sort((a, b) => a.name.localeCompare(b.name));

    } else {

        filteredProducts.sort((a, b) => b.id - a.id);

    }



    currentPage = 1;

    renderPage(1);

}



// C. Renderizar Productos (Paginados)

function renderPage(page) {

    const grid = document.getElementById('products-grid');

    const noResults = document.getElementById('no-results');

    const paginationContainer = document.getElementById('pagination');

    

    if(!grid) return;

    grid.innerHTML = '';

    

    // Calcular índices

    const start = (page - 1) * ITEMS_PER_PAGE;

    const end = start + ITEMS_PER_PAGE;

    const pageProducts = filteredProducts.slice(start, end);



    // Sin resultados

    if (pageProducts.length === 0) {

        grid.classList.add('hidden');

        if(noResults) noResults.classList.remove('hidden');

        if(paginationContainer) paginationContainer.innerHTML = '';

        return;

    } else {

        grid.classList.remove('hidden');

        if(noResults) noResults.classList.add('hidden');

    }



    // Generar Tarjetas

    pageProducts.forEach((p, index) => {

        const imagePath = p.image; // Ya viene procesado desde PHP

        

        const cardHTML = `

            <div class="group bg-white dark:bg-brand-dark border border-gray-200 dark:border-white/5 hover:border-brand-red transition-all flex flex-col h-full shadow-sm hover:shadow-neon animate-fade-in-up" style="animation-delay: ${index * 50}ms">

                <div class="relative h-64 bg-white flex items-center justify-center p-6 cursor-pointer overflow-hidden" onclick="openProductModal(${p.id})">

                    <img src="${imagePath}" onerror="this.src='public/img/no-image.png'" class="h-full object-contain group-hover:scale-110 transition-transform duration-500">

                    <span class="absolute top-3 right-3 bg-black text-white text-[10px] font-black px-2 py-1 uppercase tracking-widest">${p.brand}</span>

                    <span class="absolute bottom-3 left-3 text-[10px] text-gray-400 font-bold uppercase">${p.sub_cat}</span>

                    ${p.stock < 3 && p.stock > 0 ? '<span class="absolute top-3 left-3 bg-brand-red text-white text-[9px] font-black px-2 py-1 uppercase animate-bounce">Stock Bajo</span>' : ''}

                </div>

                <div class="p-5 border-t border-gray-100 dark:border-white/5 flex-1 flex flex-col">

                    <h3 class="font-display font-bold text-lg text-gray-900 dark:text-white uppercase mb-2 line-clamp-2 leading-tight group-hover:text-brand-red transition-colors">${p.name}</h3>

                    <p class="text-xs text-gray-500 mb-4 line-clamp-2">${p.short_description || p.description.substring(0, 60)}...</p>

                    

                    <div class="mt-auto flex justify-between items-end border-t border-gray-100 dark:border-white/5 pt-4">

                        <div>

                            <p class="text-[9px] text-gray-400 uppercase font-black tracking-wider">Precio Online</p>

                            <p class="text-2xl font-black text-brand-red italic">S/ ${p.price.toLocaleString('en-US', {minimumFractionDigits:2})}</p>

                        </div>

                        <button onclick="event.stopPropagation(); addToCart(${p.id})" class="w-10 h-10 bg-brand-black dark:bg-white text-white dark:text-black rounded-full flex items-center justify-center hover:bg-brand-red dark:hover:bg-brand-red hover:text-white transition-all shadow-lg active:scale-95">

                            <i class="ri-add-line text-xl"></i>

                        </button>

                    </div>

                </div>

            </div>

        `;

        grid.innerHTML += cardHTML;

    });



    renderPaginationControls(page);

}



// D. Controles de Paginación

function renderPaginationControls(page) {

    const totalPages = Math.ceil(filteredProducts.length / ITEMS_PER_PAGE);

    const container = document.getElementById('pagination');

    if(!container) return;

    container.innerHTML = '';



    if (totalPages <= 1) return;



    // Botón Anterior

    if (page > 1) {

        container.innerHTML += `<button onclick="goToPage(${page - 1})" class="w-10 h-10 flex items-center justify-center border border-gray-300 dark:border-gray-700 hover:bg-brand-red hover:text-white hover:border-brand-red transition-all rounded-full"><i class="ri-arrow-left-s-line"></i></button>`;

    }



    // Números de página

    let startPage = Math.max(1, page - 2);

    let endPage = Math.min(totalPages, page + 2);



    for (let i = startPage; i <= endPage; i++) {

        const activeClass = i === page ? 'bg-brand-red text-white border-brand-red shadow-neon' : 'text-gray-500 hover:text-brand-black dark:hover:text-white border-transparent';

        container.innerHTML += `<button onclick="goToPage(${i})" class="w-10 h-10 flex items-center justify-center font-bold text-sm border ${activeClass} transition-all rounded-full">${i}</button>`;

    }



    // Botón Siguiente

    if (page < totalPages) {

        container.innerHTML += `<button onclick="goToPage(${page + 1})" class="w-10 h-10 flex items-center justify-center border border-gray-300 dark:border-gray-700 hover:bg-brand-red hover:text-white hover:border-brand-red transition-all rounded-full"><i class="ri-arrow-right-s-line"></i></button>`;

    }

}



function goToPage(page) {

    currentPage = page;

    renderPage(page);

    document.getElementById('catalogo').scrollIntoView({ behavior: 'smooth', block: 'start' });

}



function resetFilters() {

    document.getElementById('search-input').value = '';

    document.getElementById('min-price').value = '';

    document.getElementById('max-price').value = '';

    const sortSelect = document.getElementById('sort-select');

    if(sortSelect) sortSelect.value = 'newest';

    

    document.querySelectorAll('.filter-category').forEach(cb => cb.checked = false);

    applyFilters();

}



// ============================================================

// 3. MODALES Y CARRITO

// ============================================================



function openProductModal(id) {
    const product = allProducts.find(p => p.id == id);
    if (!product) return;

    // Asignar datos a la vista
    document.getElementById('modal-title').textContent = product.name;
    document.getElementById('modal-brand').textContent = product.brand;
    const bgBrand = document.getElementById('modal-brand-bg');
    if(bgBrand) bgBrand.textContent = product.brand;
    
    document.getElementById('modal-description').textContent = product.description;
    document.getElementById('modal-image').src = product.image;
    document.getElementById('modal-price').textContent = product.price.toLocaleString('en-US', {minimumFractionDigits: 2});
    document.getElementById('modal-category').textContent = `${product.category} / ${product.sub_cat}`;

    // Magia PRO: El botón de WhatsApp ahora incluye el nombre del producto
    const pName = encodeURIComponent("Hola equipo JFD, me interesa cotizar este equipo que vi en su web: *" + product.name + "*");
    document.getElementById('modal-whatsapp-link').href = `https://wa.me/51989928670?text=${pName}`;

    // Evento del carrito
    const btn = document.getElementById('modal-add-cart');
    if(btn) btn.onclick = function() { addToCart(product.id); closeProductModal(); };

    // Construir la Ficha Técnica más compacta
    const specsContainer = document.getElementById('modal-specs-container');
    specsContainer.innerHTML = '';
    
    if (product.specifications) {
        try {
            const specs = JSON.parse(product.specifications);
            Object.entries(specs).forEach(([key, value]) => {
                specsContainer.innerHTML += `
                    <div class="flex flex-col border-b border-gray-200 dark:border-white/5 pb-2">
                        <span class="text-[9px] uppercase font-bold text-gray-400 tracking-wider break-words">${key}</span>
                        <span class="text-xs font-bold text-gray-800 dark:text-gray-200 whitespace-normal leading-tight mt-0.5">${value}</span>
                    </div>`;
            });
        } catch (e) { 
            specsContainer.innerHTML = '<span class="text-gray-400 italic text-xs col-span-full">Detalles generales disponibles.</span>'; 
        }
    }

    const modal = document.getElementById('product-modal');
    const content = document.getElementById('product-modal-content');
    
    // Preparar animación
    modal.classList.remove('hidden');
    modal.classList.add('flex');
    document.body.style.overflow = 'hidden'; // Bloquear scroll de fondo
    
    // Disparar animación (escala de 95% a 100%)
    setTimeout(() => {
        modal.classList.remove('opacity-0');
        modal.classList.add('opacity-100');
        content.classList.remove('scale-95', 'translate-y-4');
        content.classList.add('scale-100', 'translate-y-0');
    }, 10);
}



function closeProductModal() {
    const modal = document.getElementById('product-modal');
    const content = document.getElementById('product-modal-content');
    
    // Revertir animación
    modal.classList.remove('opacity-100');
    modal.classList.add('opacity-0');
    content.classList.remove('scale-100', 'translate-y-0');
    content.classList.add('scale-95', 'translate-y-4');
    
    // Ocultar y devolver scroll
    setTimeout(() => {
        modal.classList.remove('flex');
        modal.classList.add('hidden');
        document.body.style.overflow = '';
    }, 300);
}



// Modales Blog

function openArticleModal(articleId) {

    const data = blogPosts[articleId];

    if (!data) return;



    document.getElementById('article-image').src = data.image;

    document.getElementById('article-category').innerText = data.category;

    document.getElementById('article-title').innerText = data.title;

    document.getElementById('article-date').innerText = data.date;

    document.getElementById('article-read-time').innerText = "Lectura: " + data.readTime;

    document.getElementById('article-content').innerHTML = data.content;



    const modal = document.getElementById('article-modal');

    modal.classList.remove('hidden');

    modal.classList.add('flex');

    setTimeout(() => modal.classList.remove('opacity-0'), 10);

    document.body.style.overflow = 'hidden';

}



function closeArticleModal() {

    const modal = document.getElementById('article-modal');

    modal.classList.add('opacity-0');

    setTimeout(() => {

        modal.classList.add('hidden');

        modal.classList.remove('flex');

        document.body.style.overflow = '';

    }, 500);

}

// ============================================================
// MENÚ MÓVIL
// ============================================================
function toggleMenu() {
    const menu = document.getElementById('mobile-menu');
    menu.classList.toggle('translate-x-full');
    
    // Bloquea el scroll de la página de fondo cuando el menú está abierto
    if (!menu.classList.contains('translate-x-full')) {
        document.body.style.overflow = 'hidden';
    } else {
        document.body.style.overflow = '';
    }
}

// Carrito

function toggleCart() {

    const sidebar = document.getElementById('cart-sidebar');

    const overlay = document.getElementById('cart-overlay');

    

    if (sidebar.classList.contains('translate-x-full')) {

        sidebar.classList.remove('translate-x-full');

        overlay.classList.remove('hidden');

        setTimeout(() => overlay.classList.remove('opacity-0'), 10);

    } else {

        sidebar.classList.add('translate-x-full');

        overlay.classList.add('opacity-0');

        setTimeout(() => overlay.classList.add('hidden'), 300);

    }

}



function updateCart() {

    localStorage.setItem('jfd_cart_vfinal', JSON.stringify(cart));

    const count = cart.reduce((acc, i) => acc + i.qty, 0);

    const badge = document.getElementById('cart-badge');

    if(badge) {

        badge.innerText = count;

        badge.classList.toggle('scale-0', count === 0);

    }

    renderCartItems();

}



function addToCart(id) {

    const p = allProducts.find(i => i.id == id);

    const exists = cart.find(i => i.id == id);

    if (exists) exists.qty++; else cart.push({...p, qty: 1});

    updateCart();

    

    const sidebar = document.getElementById('cart-sidebar');

    if(sidebar.classList.contains('translate-x-full')) toggleCart();

}



function renderCartItems() {
    const container = document.getElementById('cart-items');
    const totalEl = document.getElementById('cart-total');
    if(!container) return;

    if (cart.length === 0) {
        container.innerHTML = '<div class="flex flex-col items-center justify-center h-full text-center text-gray-400 space-y-4"><i class="ri-shopping-cart-line text-4xl opacity-20"></i><p class="text-sm">Tu arsenal está vacío.</p></div>';
        totalEl.innerText = 'S/ 0.00';
        return;
    }

    const total = cart.reduce((acc, i) => acc + (i.price * i.qty), 0);
    totalEl.innerText = `S/ ${total.toFixed(2)}`;

    container.innerHTML = cart.map(i => `
        <div class="flex gap-4 items-center bg-white dark:bg-white/5 p-3 rounded-lg border border-gray-200 dark:border-white/5 shadow-sm transition-colors">
            <div class="w-16 h-16 bg-white rounded-md flex items-center justify-center p-1 border border-gray-100 shrink-0 overflow-hidden">
                <img src="${i.image}" 
                     onerror="this.src='public/img/no-image.png'" 
                     class="w-full h-full object-contain">
            </div>
            <div class="flex-1 min-w-0">
                <h4 class="text-xs font-bold uppercase text-gray-900 dark:text-white line-clamp-2 leading-tight mb-1">${i.name}</h4>
                <div class="flex justify-between items-center mt-2">
                    <span class="text-brand-red font-black text-sm">S/ ${(i.price * i.qty).toFixed(2)}</span>
                    <div class="flex items-center gap-1 bg-gray-100 dark:bg-black/40 rounded px-1 py-0.5 border border-gray-200 dark:border-white/10">
                        <button onclick="modQty(${i.id}, -1)" class="w-6 h-6 flex items-center justify-center text-gray-500 hover:text-brand-red font-bold">-</button>
                        <span class="text-xs font-bold text-gray-900 dark:text-white w-4 text-center">${i.qty}</span>
                        <button onclick="modQty(${i.id}, 1)" class="w-6 h-6 flex items-center justify-center text-gray-500 hover:text-green-500 font-bold">+</button>
                    </div>
                </div>
            </div>
        </div>
    `).join('');
}



function modQty(id, delta) {

    const item = cart.find(i => i.id === id);

    if (item) {

        item.qty += delta;

        if (item.qty <= 0) cart = cart.filter(i => i.id !== id);

        updateCart();

    }

}



function checkout() {

    if (cart.length === 0) return alert("Agrega productos primero.");

    let msg = "Hola JFD Importaciones! 🎧 Deseo cotizar:\n\n";

    cart.forEach(i => msg += `▪ ${i.qty}x ${i.name} (S/ ${i.price})\n`);

    msg += `\n*TOTAL ESTIMADO: S/ ${cart.reduce((a,b)=>a+(b.price*b.qty),0).toFixed(2)}*`;

    window.open(`https://wa.me/51989928670?text=${encodeURIComponent(msg)}`, '_blank');

}



// ============================================================
// 4. INICIALIZACIÓN GLOBAL (DOM READY)
// ============================================================

function initTheme() {
    const html = document.documentElement;
    const themeBtn = document.getElementById('theme-btn');
    const headerLogo = document.getElementById('header-logo');
    
    // 1. Verificar preferencia guardada o usar 'dark' por defecto
    const savedTheme = localStorage.getItem('theme') || 'dark';
    
    // 2. Aplicar clase inicial
    if (savedTheme === 'dark') {
        html.classList.add('dark');
    } else {
        html.classList.remove('dark');
    }

    // 3. Lógica del Logo (Asegúrate de tener la imagen 'logo-jfd-dark.png' si usas modo claro)
    const updateLogo = (isDark) => {
        if(headerLogo) {
            // Si no tienes logo oscuro, usa el mismo para ambos
            headerLogo.src = isDark ? 'public/img/logo-jfd.png' : 'public/img/logo-jfd-dark.png'; 
        }
    };
    
    updateLogo(savedTheme === 'dark');

    // 4. Evento Click (Usamos onclick para evitar duplicidad de listeners)
    if (themeBtn) {
        themeBtn.onclick = () => {
            const isDarkNow = html.classList.toggle('dark');
            localStorage.setItem('theme', isDarkNow ? 'dark' : 'light');
            updateLogo(isDarkNow);
        };
    }
}



// Inicialización Maestra

document.addEventListener('DOMContentLoaded', () => {

    initTheme();

    renderCategoryFilters(); // Generar Filtros de Sidebar

    renderPage(1);           // Mostrar productos

    updateCart();            // Cargar carrito

    

    // Botón Back to Top

    const backToTopBtn = document.getElementById('back-to-top');

    if (backToTopBtn) {

        window.addEventListener('scroll', () => {

            if (window.scrollY > 500) {

                backToTopBtn.classList.remove('opacity-0', 'translate-y-10', 'pointer-events-none');

            } else {

                backToTopBtn.classList.add('opacity-0', 'translate-y-10', 'pointer-events-none');

            }

        });

        backToTopBtn.addEventListener('click', () => window.scrollTo({ top: 0, behavior: 'smooth' }));

    }

});


// Inicialización Maestra
document.addEventListener('DOMContentLoaded', () => {
    initTheme();             // Tema
    renderCategoryFilters(); // Filtros Sidebar
    renderPage(1);           // Productos
    updateCart();            // Carrito
    
    // Botón Back to Top
    const backToTopBtn = document.getElementById('back-to-top');
    if (backToTopBtn) {
        window.addEventListener('scroll', () => {
            if (window.scrollY > 500) {
                backToTopBtn.classList.remove('opacity-0', 'translate-y-10', 'pointer-events-none');
            } else {
                backToTopBtn.classList.add('opacity-0', 'translate-y-10', 'pointer-events-none');
            }
        });
        backToTopBtn.addEventListener('click', () => window.scrollTo({ top: 0, behavior: 'smooth' }));
    }
});

// --- CÓDIGO TEMPORAL PARA ARREGLAR EL CARRITO ---
// Si detecta que no hemos limpiado el caché hoy, borra el carrito viejo
if (!sessionStorage.getItem('cart_fix_applied')) {
    console.log("Limpiando carrito con rutas viejas...");
    localStorage.removeItem('jfd_cart_vfinal'); // Borra la memoria del carrito
    sessionStorage.setItem('cart_fix_applied', 'true'); // Marca que ya se hizo
    window.location.reload(); // Recarga la página automáticamente
}
// -------------------------------------------------
</script>
</body>
</html>